#Loading the necessary libraries
using Plots
using ModelingToolkit
using DifferentialEquations
using Random
using Statistics
using OrdinaryDiffEq
using Lux 
using DiffEqFlux
using Flux
using ComponentArrays 
using Optimization, OptimizationOptimJL, OptimizationOptimisers                                                                   
rng = Random.default_rng()
Random.seed!(99)

#No Noise forecasting from eta = 0 to the 90th element of the domain etaspan = (0.05, 5.325)


#Constants
C = 0.01


#Initial Conditions
I = [1.0, 0.0]   #Psi(0)=1, Psi'(0)=1
etaspan = (0.05, 5.325)

#radius range
datasize= 100
etasteps = range(etaspan[1], etaspan[2]; length = datasize)


function whitedwarf(du, u, p, r)
    psi = u[1]
    dpsi = u[2]
    du[1] = dpsi
    du[2] = (-((psi^2-C))^(3/2) - 2/r * dpsi)
end


#Defining the Ordinary differential equation as an ODEProblem with the DifferentialEquations.jl
prob = ODEProblem(whitedwarf, I, etaspan)
#Solving the ODEProblem with the Tsit5() algorithm
sol = solve(prob,saveat=etasteps)

#Plot
plot(sol, linewidth = 1, title = "White Dwarf equation", xaxis = "\\eta",
     label = ["\\phi" "\\phi'"])

#--------------I will solve the white dwarf equation using the SecondOrderODEProblem function------------

#Defining the function containing the Second Order Differential Equation
function whitedwarf2(ddu,du,u,C,eta)
    ddu .= (-((u.*u.-C)).^(3/2) - 2/eta * du)
end

#Initial conditions definined as required by the syntax of the Second Order Differential Equation
dpsi0=[0.0]
psi0=[1.0]
#Defining the secondOrderProblem 
prob2 = SecondOrderODEProblem(whitedwarf2,dpsi0, psi0, etaspan, C)
#Solving it with the automated choosen algorithm
sol2 = solve(prob2, saveat=etasteps)

#plot sol2
plot(sol2, linewidth=1.5, title = "White Dwarf equation", xaxis = "\\eta", label = ["\\phi" "\\phi '"])


#-------------------------------------Defining the Neural ODE------------------------------------


dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)

etasteps =  etasteps[1:end-10]
etaspan = (etasteps[1], etasteps[end])
I=[1.0,0]
prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)

function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end
#Training data
true_data= Array(sol[:,1:end-10])
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end


callback = function (p, l, pred; doplot = true)
    println(l)
    # plot current prediction against data
    if doplot

        plt1 = scatter(collect(etasteps), true_data[1, :]; label = "\\phi data")
        scatter!(plt1, collect(etasteps), pred[1, :],markershape=:xcross; label = "\\phi prediction")
        scatter!(plt1, collect(etasteps), true_data[2, :]; label = "\\phi ' data")
        scatter!(plt1, collect(etasteps), pred[2, :],markershape=:xcross; label = "\\phi ' prediction")
        #plt1 = scatter(sol.t, true_data[3, :]; label = "data")
        #scatter!(plt1, sol.t, pred[3, :]; label = "prediction")
        #plt=plot(plt1, plt2)
        
        display(plot(plt1))

        
        
    end
    return false
end


pinit = ComponentArray(p)
callback(pinit, loss_neuralode(pinit)...; doplot = true)




# use Optimization.jl to solve the problem
adtype = Optimization.AutoZygote()

optf = Optimization.OptimizationFunction((x, p) -> loss_neuralode(x), adtype)
optprob = Optimization.OptimizationProblem(optf, pinit)

result_neuralode = Optimization.solve(optprob, OptimizationOptimisers.Adam(0.1); callback = callback,
    maxiters = 80)

optprob2 = remake(optprob; u0 = result_neuralode.u)

result_neuralode2 = Optimization.solve(optprob2, Optim.BFGS(; initial_stepnorm = 0.01);
    callback, allow_f_increases = false, maxiters=100)

callback(result_neuralode2.u, loss_neuralode(result_neuralode2.u)...; doplot = true)

xlabel!("Eta (dimensionless radius)")
title!("Trained Neural ODE vs Ground truth data")

savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0\\Neural ODE\\Results\\NoNoise\\Whitedwarf_TrainedNeuralODEvstrainingdata.png")



#Saving the trained parameters
open("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0\\Neural ODE\\Trained_parameters\\p_minimized_nonoise.txt","w") do f

    write(f, string(result_neuralode2.minimizer))
end
#---------------------Forecasting-----------------------#
#------------------------------------------------------

#------------------------------------------------------


function dudt_node(u,p,t)
    phi, phiderivative = u
   
    output, _ = dudt2([phi,phiderivative],p,st)
    dphi, dphiderivative = output[1],output[2]
    return [dphi,dphiderivative]
end

#Initial Conditions
I = [1.0, 0.0]   #Psi(0)=1, Psi'(0)=1
etaspan2 = (0.05, 5.325)

#radius range
datasize= 100
etasteps2 = range(etaspan2[1], etaspan2[2]; length = datasize)

p_trained=(layer_1 = (weight = Float32[-0.5836619 0.4376292; -0.8438847 -0.63573873; -0.22818492 -0.16087975; -0.058397036 -1.2134186; -0.66212684 0.6330059; -0.6018261 -0.83270854; 0.4901169 -0.39823735; 0.33831456 -0.53536886; -1.1017618 0.24213287; 0.15590152 1.5958364; -0.8491539 -0.049485054; 0.58421326 -0.62068045; 0.27540046 -0.024816215; 0.1388686 -1.1266612; -0.3256207 0.067228585; 0.8663603 -0.36634585; 0.5500781 -0.07733675; 0.64877224 -0.83104366; 0.4919355 0.7158439; 0.06852026 0.68308914; -0.34694472 0.12416278; 0.6570016 -0.0932242; -0.87470794 -0.16549127; 0.5981263 -0.123304054; 0.7964552 -0.15019432; 0.06314678 0.80510193; -0.641391 -0.0006132077; 0.16636837 1.1728938; -0.07892441 -0.11649985; 0.9746405 0.67424834; 0.4482575 0.6243109; -0.7161666 -0.21974187; 0.7785012 -0.21918012; -0.89574087 0.743669; -0.4264137 -1.4958879; 0.6460656 -0.52937335; -0.60968894 0.12614328; 0.30978617 1.3631048; 0.6447419 0.14467306; -0.7083478 -1.5734134; -0.5764454 0.3449271; -0.7420932 0.031896327; -0.115757786 -0.68838435; -0.7848534 0.044804778; -0.47406423 1.1070065; 0.60586345 -0.4251131; -0.92915577 0.7042489; -0.43897098 -0.6338367; 0.40875518 -0.52939004; 0.36264953 -0.11258128; -1.0538148 0.36766702; 0.8031039 -0.24348474; -0.4483093 -0.930856; 0.72940296 0.13522026; -0.5192829 -0.06552727; -0.40019205 0.005540917; 0.8011548 0.04203193; -0.8637946 0.23198454; -0.9509497 0.12896462; 0.5303764 1.2414685; 0.47005555 0.76475227; -0.630275 0.26209795; 0.5185801 -0.57950395; -0.29712626 0.697611; -0.54864514 -0.696941; 0.7915787 0.35139608; 1.0143443 -0.82775736; 1.0244046 0.2786547; -1.032274 0.51786345; 0.8747835 0.53835523; 1.071358 -0.24084942; 0.77858096 -0.67578053; -0.7707391 0.20725952; -0.21004583 -0.5400134; -0.07694214 -1.5469654; 0.54887563 1.4507291; -0.27408078 0.26656792; 0.43058622 0.951756; -0.5737836 0.09376381; -0.41529992 -0.6823074], bias = Float32[-0.7261357; -0.3940102; 0.43703058; -0.31599912; -0.69152457; -0.23458612; 0.89062893; 0.9270689; -0.81647927; 0.018953865; -0.74108577; 0.6475313; -0.21473932; -0.2290055; -0.965899; 0.8197715; 0.83668983; 0.514139; 0.18303564; -0.5864642; 0.56615776; 0.042237546; -0.9466446; 0.83810353; 0.93466127; -0.20967546; -0.92141837; -0.025643019; -0.53786623; 0.8833189; -0.02521705; -0.8925286; 0.69938093; -0.67027867; -0.37881997; 0.7953787; -0.93308395; -0.21105576; 0.74155873; -0.39467135; -0.59104; -0.89519167; 0.30798772; -0.9387116; -0.06764954; 0.82132787; -0.81434244; 0.184269; 0.76518285; 0.22247554; -0.7661458; 0.86382854; -0.123442285; 0.9453286; -0.09463076; -0.72832555; 0.97155446; -0.78622156; -0.83431476; 0.47445178; 0.68112254; -0.8429541; 0.6456532; -0.44352978; -0.68279034; 0.67084366; 0.39983174; 1.3204399; -0.6879069; 0.18245637; 0.71518546; 0.7950578; -0.8961277; 0.24540922; 0.20833266; 0.110288486; -0.87119687; -0.05110301; -0.8877154; -0.35762513]), layer_2 = (weight = Float32[-0.79266834 0.8982822 0.683286 -0.19177446 -0.84237576 0.92468953 0.66386455 -0.7834701 0.9440418 0.5413672 -0.13743044 0.6847691 -0.7845979 0.34739718 0.51500857 -0.6161726 0.12717547 0.41288126 0.748182 0.7763306 -0.10911831 -0.047705993 -0.16911782 0.5151288 -0.95107514 0.79908144 -0.28731057 0.46907642 -0.3214008 -0.9232973 0.84681875 0.5083256 0.4989304 0.027047165 0.20248392 0.8733633 -0.28274122 0.44324556 0.21719253 -0.03016438 -0.3515198 0.23113358 0.6652752 0.4115307 -0.14274842 0.6502272 0.16248849 -0.8009326 0.8883013 -0.33159807 0.67132527 0.63457555 -0.79855406 -0.43863147 0.8381644 -0.27044213 -0.65777683 -0.02416389 0.6667091 -0.5083465 -0.34066948 -0.7424268 0.17013638 0.778118 0.17777726 -0.9555539 -0.8249071 -0.7423909 0.90688854 -0.66421086 -0.8876557 -0.37349677 1.0497975 0.6997214 -0.25254595 0.044331294 0.34971964 0.79002184 -0.6227599 -0.5657342; -0.3763834 0.40887463 0.46208796 -0.74317646 -0.6352258 0.549208 0.382233 -0.21098301 0.46133983 0.7378652 0.5847139 0.07707151 -0.48809594 -0.6912047 -0.6812638 -0.48196942 0.7072162 -0.76654917 0.520398 0.46249506 0.7602266 -0.518851 0.67542243 0.47778824 -0.71824193 0.5809745 -0.5317201 0.525267 -0.41171235 -0.65009284 0.40553316 -0.55376667 0.5419663 0.71300155 -0.7026964 0.37700957 -0.40525272 0.5907165 -0.81901973 -0.7995879 0.84624726 -0.3653502 0.2920658 -0.4310295 0.65888494 0.27782825 0.7536615 -0.38581234 0.44727796 -0.6909712 0.4711024 0.44308278 -0.41371167 0.70334464 0.49249333 -0.4718861 -0.8123032 -0.6742238 -0.28218645 0.736466 0.5152123 -0.59636647 -0.9988611 0.7163541 -0.7132917 -0.67797554 -0.50070995 -0.5631543 0.8331012 -0.69570255 -0.71487653 -0.69238955 0.699226 0.40176663 -0.57995856 0.623589 -0.5458172 0.67747915 -0.6487877 -0.6820929; -0.25924093 0.44254187 -0.20773046 0.7178975 -0.3574947 0.44895536 0.5065821 -0.44979784 0.2512776 -0.71435684 -0.6263768 0.45250192 0.29216358 0.53889567 0.4710979 -0.49200404 -0.89754 0.5074357 0.3938783 0.25049683 -0.8406282 0.66218984 -0.6353346 -0.79521996 -0.52814436 0.39861614 0.6654418 -0.62931496 0.6137355 -0.3028369 0.27163753 0.53318316 -0.73683286 -0.84966254 0.5036116 0.15694344 0.7441524 -0.84178823 0.6899143 0.71683174 -0.50412107 0.82909846 0.46550882 0.60777634 -0.69135493 0.24658892 -0.86789316 -0.3216717 0.20548122 0.59256375 0.2425316 -0.28713483 -0.25307223 -0.8444345 -0.26909974 0.74086785 -0.06209254 0.8235056 0.5084485 -0.7187826 -0.5639001 0.28713748 0.75172156 0.22086458 0.6167889 -0.28810012 -0.38854307 0.6967916 0.1164651 -0.47555956 -0.29578543 0.53721094 0.5466392 0.36502042 0.67906934 -0.67384464 0.5773839 0.24062392 0.28775942 0.28417775; 0.35908863 0.064978816 -0.30077252 0.91085035 0.011510708 0.38050392 -0.40212044 -0.6033204 -0.15099578 -0.6776714 -0.8678908 0.3201534 0.61625063 0.71180016 0.5895463 -0.41612366 -0.81335974 0.6688695 0.01683032 0.37275267 -0.82699424 0.84014624 -0.83400226 -0.92182285 -0.5567657 -0.29612342 1.0584775 -0.54851913 0.8299802 0.033435136 0.36615536 0.77505875 -0.6229533 -0.9076923 0.8057765 -0.35489976 1.0534741 -0.594249 0.61535966 0.83859575 -0.74451584 1.0095638 -0.05768032 1.116828 -0.7792205 0.11396727 -0.53614265 -0.11284376 -0.06454041 0.6967713 0.01840929 -0.7086907 0.17184418 -1.0503287 -0.37298745 0.9195025 0.38077816 0.8976505 0.4532196 -0.8925928 -1.09238 0.8012004 0.73167264 0.10099526 0.9710734 0.23315543 0.014558195 0.36871836 -0.16480902 -0.2783111 -0.18771192 0.79620796 0.0008546052 -0.15133215 0.89238787 -0.9102442 0.6529669 -0.2856712 0.704878 0.56956565; 0.20013534 -0.15423256 0.27247137 -0.9626821 0.17761098 -0.44117957 -0.28085917 0.461495 -0.22172977 0.820093 0.78077644 -0.60436195 -0.16854914 -0.70638597 -0.71839094 0.30255437 0.8463073 -0.59080696 -0.42145753 -0.396951 0.8532398 -0.6491329 0.78171515 0.9755145 0.4542573 0.04771711 -0.7560016 0.5812428 -0.65018904 0.23533826 -0.47489366 -0.74214554 0.8388968 0.8317599 -0.90797347 -0.09374936 -0.9122831 0.74830073 -0.8819443 -0.7705304 0.69857186 -0.72200805 -0.4709571 -0.8022739 0.84287804 -0.40485814 0.97219235 0.43274754 -0.17779559 -0.84889513 0.053797744 0.51040405 0.30908665 0.7805656 0.22588198 -0.6430882 -0.031658765 -0.66512 -0.40233493 0.6760746 0.6316233 -0.2955824 -0.6885016 -0.110262774 -0.7846608 0.12319472 0.30629608 -0.6734405 0.10472512 0.21279922 0.37561917 -0.80405027 -0.1697255 -0.055192098 -0.8351311 0.8997864 -0.5148004 -0.13036884 -0.50796396 -0.27179092; 0.94196564 -0.9340294 -0.84187037 0.17404401 0.6238631 -0.56625605 -0.5929464 0.7527138 -0.741584 -0.41610792 0.19498047 -0.48921818 0.64426726 -0.32311186 -0.390994 0.8752898 -0.09669651 -0.24759994 -0.5774405 -0.9719702 0.47048935 -0.11208471 -0.054402422 -0.2426355 0.7965469 -1.0475551 -0.063069716 -0.7228623 0.44536564 1.0779786 -0.77259356 -0.41242722 -0.33816785 0.09610638 -0.49917257 -0.63307273 -0.05949957 -0.7499939 -0.31770036 0.010149363 0.2533542 -0.22306342 -0.87748796 -0.39564538 0.053912893 -0.9006164 -0.16885227 0.66439986 -0.68652284 0.3322747 -1.0466176 -0.61707246 0.7433787 0.49938577 -0.71104664 0.39389816 0.67698073 -0.120500885 -0.90127724 0.4936587 0.124144375 0.51858103 -0.3299439 -0.8465107 -0.31350836 0.9220722 1.0652468 0.87468094 -0.9165216 0.7139456 0.74714506 0.067357115 -0.80288947 -0.92005163 0.20572479 -0.034096163 -0.32987112 -0.9381263 0.48818278 0.6491199; 0.811286 -0.74256754 -0.7892444 -0.5820828 0.58511126 -0.7600566 -0.7265882 0.66259277 -0.55379915 0.21894342 0.5856006 -0.85354954 0.6816785 -0.43959087 -0.45537075 0.43536445 0.11004968 -0.5387672 -0.6857896 -0.6240857 0.5585973 -0.29888153 0.49156398 0.20377412 0.6832422 -0.52151394 -0.43340102 -0.35052675 -0.223797 0.5665197 -0.83818305 -0.40762383 -0.21502842 0.3935712 -0.5212256 -0.8136149 -0.10352612 -0.3567737 -0.57438916 -0.15875474 0.69163483 -0.3705757 -0.60708266 -0.2378014 0.46050853 -0.8214027 0.35010716 0.59350634 -0.6490314 -0.4413597 -0.7043104 -0.6986497 0.86881036 0.25951925 -0.84096074 -0.42056647 0.7314141 -0.14228892 -0.42741683 0.4125611 0.25631258 1.0863637 -0.58367145 -0.70696694 -0.5265089 0.61097836 0.53941953 0.68959844 -0.578714 0.41993323 0.44764116 -0.3048788 -0.6185719 -0.88058096 -0.2136968 0.4609817 -0.5900485 -0.73886025 0.96111405 0.7773359; -0.7023651 0.49394253 0.7458105 0.44090766 -0.6758221 0.6447407 0.516143 -0.43357453 0.52835023 -0.44830114 -0.59450203 0.32749316 -0.8577107 0.5220305 0.4027598 -0.5296477 -0.58061665 0.6115194 0.5583484 0.83544016 -0.2821237 0.3778768 -0.32502356 -0.22639498 -0.80807483 0.6576907 0.5061975 0.021612734 0.2395777 -0.7271966 0.5035506 0.5946509 -0.15493673 -0.5899216 0.44916284 0.802457 0.6118462 -0.16432025 0.46230853 0.40618154 -0.26452556 0.32933602 0.69464964 0.63748455 -0.4927383 0.8411063 -0.5341919 -0.66666466 0.808009 0.2624096 0.56536674 -0.10188518 -0.490251 -0.27342218 0.86398214 0.37450007 -0.5584335 0.3729137 0.39406177 -0.44586796 -0.6379334 -0.8280496 0.325416 0.5837375 0.52101064 -0.8321372 -0.8374249 -0.97054315 0.74416316 -0.57779455 -0.7413316 0.58724207 0.535506 0.7574253 0.5400291 -0.53891486 0.50583977 0.5577157 -0.61470217 -0.7277825; -0.69013155 0.6851956 0.60152334 -0.4662764 -0.75684714 0.9386737 0.60133153 -0.89574254 0.5795417 0.5824998 0.60448027 0.79037297 -0.68672913 -0.5928849 0.0015920943 -0.73748434 0.60168767 -0.4460639 0.7743621 0.6185754 0.31886724 -0.5496878 0.5976711 0.5702348 -0.8912196 0.6637803 -0.37593764 0.74694234 -0.46139714 -0.5761355 0.6011916 -0.42505828 0.38749284 0.43028513 -0.26442856 0.515993 -0.4429383 0.6873748 -0.29610327 -0.40380988 0.3951082 -0.30541506 0.69289035 -0.43790692 0.4363431 0.52048856 0.76421875 -0.65884674 0.68058187 -0.6266958 0.74704134 0.6695314 -0.537459 0.16575515 0.5913043 -0.72692156 -0.79954094 -0.5356883 0.8281281 0.28448987 0.36017385 -0.59101504 -0.35658026 0.7350999 -0.47391072 -0.5642985 -0.5083944 -0.4548245 0.6858987 -0.76707953 -0.6736578 -0.41604 0.61721325 0.5617577 -0.67083347 0.6255261 0.7194963 0.44077802 -0.62549025 -0.56105745; -0.2196951 0.01624216 0.67255324 -0.8703183 0.050184976 -0.18564636 0.05223671 0.2930583 0.25625837 0.723773 0.8941515 -0.4726175 -0.46976474 -0.8297759 -0.8788901 0.36002493 0.64795125 -0.7911068 0.02356706 -0.29865053 0.60493124 -0.9406727 0.785228 0.60944736 0.3932692 0.07425666 -0.6340549 0.6161636 -0.8520034 -0.25199133 -0.19508696 -0.93714994 0.5352234 0.7019591 -0.6281292 0.15568079 -0.7832622 0.59852403 -0.73504823 -0.9447867 0.9120872 -0.89256865 0.047008496 -0.9084992 0.7913351 0.05176017 0.6829796 0.3090305 0.079087034 -0.90267676 0.06789711 0.8315652 -0.16467576 0.77036583 0.6493144 -0.86409384 -0.4159052 -0.95872974 -0.28050378 0.60781986 0.97546446 -0.6553016 -0.88219887 0.110334314 -0.7753005 -0.21496114 -0.19787855 -0.5358593 0.38786265 0.14128807 -0.015883535 -0.9698468 0.080409326 0.05397161 -0.76182985 0.76248825 -0.29341957 0.005772044 -0.6691953 -0.6369765; -0.26466626 0.36245248 -0.29148108 0.9433843 -0.50527096 0.4704472 0.1918772 -0.5258969 0.34119195 -0.96299857 -0.75215083 0.62618494 0.5604044 0.7291894 0.61703634 -0.22684382 -0.69902116 0.83934265 0.19175534 0.3822871 -0.68606466 0.85731727 -0.7712952 -0.6665836 -0.379114 -0.019804168 0.6726047 -0.72771597 0.7808259 -0.38094008 0.55170053 0.7703092 -0.53510714 -0.7060089 0.68807447 0.32422346 0.7807493 -0.6327151 0.60528165 0.9419818 -0.7548445 0.637923 0.52026874 0.757219 -0.9246327 0.49005532 -1.016428 -0.48857853 0.3127737 0.8453722 0.019239273 -0.31618932 -0.1934141 -0.8522117 -0.12485898 0.90730786 0.17717202 0.7775401 0.502048 -0.79240686 -0.868564 0.42434442 0.7908836 0.3793596 0.8162232 -0.26694077 -0.15636766 0.66341823 -0.12090296 -0.51561975 -0.31926635 1.0234987 0.34052458 0.14929762 0.71752787 -0.8988897 0.40499368 0.14280751 0.37380427 0.49118367; 0.2892747 -0.46737224 0.16436888 -0.8315492 0.28715774 -0.574884 -0.3872333 0.37975153 -0.2844913 0.66909575 0.6777241 -0.41041768 -0.35505378 -0.5931622 -0.84223735 0.3705241 0.93713665 -0.61062104 -0.34926897 -0.36502942 0.6924644 -0.8242485 0.81753504 0.7721409 0.38604134 -0.365332 -0.693535 0.47963288 -0.85104626 0.34390733 -0.28911284 -0.61900795 0.7136197 0.62878865 -0.6771863 -0.18043762 -0.8271187 0.7500733 -0.8737414 -0.82810414 0.8061372 -0.80618787 -0.40839514 -0.5966192 0.72947603 -0.21234207 0.8261844 0.5589399 -0.23209944 -0.85536104 -0.35160875 0.36455053 0.45832944 0.57702875 0.35441813 -0.85591 -0.17015477 -0.7283898 -0.38465837 0.5189828 0.6858227 -0.3287294 -0.8793633 -0.3230516 -0.5553597 -0.004121194 0.11389598 -0.45606568 0.12611033 0.44255662 0.5136608 -0.9030792 -0.35717967 -0.45515728 -0.7410973 0.6954147 -0.81117445 -0.16995977 -0.31619757 -0.282597; -0.61492544 0.9889612 0.6498457 -0.38123617 -0.6826282 0.7843728 0.6726313 -0.7270475 0.9797481 0.42443752 0.12188233 0.65409774 -0.8225079 0.04685782 0.3010811 -0.835087 0.3594543 -0.023052132 0.87174904 0.8649264 0.058062054 -0.26636067 0.4047648 0.64746 -0.82818484 0.90711516 -0.42121205 0.7533909 -0.46203566 -0.965601 0.9092148 0.16269329 0.692278 0.4240036 -0.105243996 0.6775032 -0.51007247 0.6385853 -0.017102508 -0.34013325 -0.18705153 0.004730537 0.9597867 -0.054126363 0.50592613 0.93936616 0.5648028 -0.96982014 0.58148044 -0.52110577 0.820134 0.47269106 -0.935561 -0.37320927 0.7676068 -0.64592737 -0.57928026 -0.19329631 0.67929864 -0.10103881 0.22427084 -0.65753806 0.118274 0.8688041 -0.1287761 -0.6761837 -0.68779105 -0.8206171 0.6637202 -0.61300343 -0.847489 -0.45477527 0.99996275 0.9015937 -0.5494333 0.28316605 0.5519238 0.7408272 -0.45934936 -0.66084576; -0.8468455 0.64078623 0.7017411 0.44326437 -0.70924485 0.81550646 0.4737477 -0.65844166 0.8525565 0.21587706 -0.27927738 0.6624102 -1.0283786 0.3219314 0.61154836 -0.94299626 -0.46438017 0.4522007 0.51013595 0.6286205 -0.32427984 0.43835044 -0.4523267 -0.15359469 -0.6784529 0.694572 0.50516945 0.5031376 0.123904444 -0.87807184 0.66479653 0.6993852 0.07457677 -0.08084472 0.48430818 0.8074476 0.43316513 0.3598636 0.1530251 0.39512032 -0.21042518 0.61292386 0.59869266 0.44050416 -0.18721765 0.4048067 -0.014355707 -0.62196934 0.7015597 0.13815734 1.0731317 0.16935234 -0.844235 -0.51659024 0.99078196 0.24032854 -0.9076517 0.575895 0.43292996 -0.6614148 -0.48068753 -0.6709525 0.41460326 0.68895227 0.5183467 -0.8046122 -0.9027399 -1.0349087 1.0269976 -0.59892935 -0.7063209 0.03733563 0.89739895 0.73167294 0.11140863 -0.27554727 0.62811637 0.8107952 -0.63014096 -0.90069234; 0.583483 -0.59346414 -0.6813817 0.5444203 0.5649822 -0.3876102 -0.60358435 -0.23412897 -0.45174596 -0.7529917 -0.6104601 -0.3815427 0.54413646 0.64089346 1.1335897 0.06831734 -0.8647148 0.65135694 -0.5382214 -0.38909543 -0.6338751 0.7265133 -0.40029436 -0.5630699 0.34191945 -0.36500785 0.69581693 -0.6987222 0.55976003 0.5921772 -0.36332652 0.9479782 -0.69961256 -0.4831591 0.7929173 -0.65642023 0.80363804 -0.532287 0.5154224 0.48910677 -0.5152607 0.66258824 -0.52189916 0.65502435 -0.6377927 -0.6216071 -0.56568944 0.34719044 -0.5534708 0.6383347 -0.55903053 -0.8981216 0.494518 -1.0268514 -0.47187117 0.65942186 0.57639027 0.91956776 0.48696715 -0.9539545 -0.8081477 0.7873998 0.680825 -0.47614154 0.765844 0.65354306 0.36214834 0.5924034 -0.35838214 0.58206457 0.5051659 0.6691017 -0.5285287 -0.6021822 0.7984824 -0.62018675 0.4549945 -0.7455733 0.81174356 0.6949888; 0.6732382 -0.6685349 -0.72670853 0.34160468 0.65508133 -0.6925833 -0.58924615 0.81861985 -0.88355315 -0.7160209 -0.08638156 -0.47890088 0.4974601 -0.34623834 -0.29576567 0.810288 -0.4346195 -0.3891182 -0.6220149 -0.81665254 0.23489553 -0.12843247 0.11900723 -0.58480287 0.7064961 -0.6891112 0.2417201 -0.723794 0.2989454 0.96480316 -0.555351 -0.47450885 -0.44800085 -0.008987202 -0.33278412 -0.64114165 0.17602505 -0.6096524 -0.015185045 -0.0025414794 0.06942211 -0.1379035 -0.7821809 -0.09835046 -0.19093846 -0.64733094 -0.23722734 0.88973993 -0.61305404 0.24296328 -0.7188553 -0.5668381 0.9813974 0.60161513 -0.81634873 0.2303616 0.9921268 -0.0006330296 -0.60560596 0.27249092 0.10108495 0.6100441 -0.43367395 -0.78953147 -0.30607757 0.83437574 0.9299542 0.7653012 -0.838822 0.7761148 1.0774109 0.36489624 -1.0209408 -0.9330097 0.451595 -0.37478414 -0.64408845 -0.8931787 0.414842 0.44484556; 1.0455412 -0.84677285 -0.7177231 0.19784254 1.1600443 -0.8466829 -0.9707171 0.43273714 -0.4817224 -0.3311336 0.04873681 -0.75193715 0.45116112 -0.2940231 -0.4003189 0.5294363 -0.4037322 -0.51032066 -0.7921952 -0.75271624 0.023067938 0.2016008 0.028498877 -0.7780155 0.74591494 -0.6306497 0.620734 -0.78129435 0.6296565 0.5850034 -0.9645166 -0.04313887 -0.8115802 -0.22209239 0.18492016 -1.1028601 0.6976107 -0.45140716 -0.30989087 0.21619755 0.37390456 0.37079787 -0.9659068 0.31505463 0.09713636 -1.0158317 -0.09191991 0.95824623 -1.1294426 0.28100893 -0.57963 -0.8424699 0.8001475 0.058300134 -0.68499875 0.5458146 0.7293016 0.2247581 -0.54514223 0.22370231 0.014690899 0.6392968 -0.299851 -0.8273204 -0.004360346 0.7233222 0.7866265 0.39864185 -0.64697224 0.8305168 0.63095653 0.18555993 -0.75123554 -0.7090816 0.43928754 -0.1550224 -0.6203837 -0.82662886 0.8964424 0.653167; 0.9042608 -0.73368734 -0.76849973 -0.35052145 1.0698844 -0.74781525 -1.0642163 0.44775748 -0.48588082 -0.346188 0.41733357 -0.91716325 0.6979627 -0.5465576 -0.5529806 0.5565041 -0.11509145 -0.6554241 -0.90777653 -0.7492656 0.24538255 -0.22913705 0.43055004 -0.30197978 0.5655685 -0.8356814 0.25244123 -0.8108222 0.32037354 0.66332513 -0.971979 -0.13361968 -0.7391683 0.35062346 -0.36949527 -0.95112467 0.31047532 -0.57325494 -0.4621355 -0.22872826 0.43356523 -0.1142434 -0.9442651 -0.004865409 0.42564023 -0.98437774 0.14775553 0.70570517 -1.0492177 -0.24546306 -0.675839 -0.7593787 0.84433734 0.25330123 -0.70538884 0.3543665 0.541537 -0.23236713 -0.5209759 0.10706409 0.15848528 0.8270579 -0.68512213 -0.6235545 -0.19883513 0.74205744 0.5516247 0.6381529 -0.79480416 0.93179864 0.65459716 -0.27382222 -0.51194614 -0.8325302 0.4346059 0.022805855 -0.44633487 -0.75654566 0.7528324 0.72264457; 0.5120606 -0.8180693 -0.58707154 0.4170513 0.6618278 -0.7103227 -0.6215334 0.70211977 -0.8573516 -0.4193191 -0.42860514 -0.90287167 0.4987267 0.48523116 -0.38097373 0.80546093 -0.56714404 0.23989762 -0.5899129 -0.7009211 -0.3039531 0.67509675 -0.44462138 -0.5801357 0.959771 -0.6293586 0.42479625 -0.4194888 0.46322063 0.6288905 -0.88970923 -0.012811836 -0.72356886 -0.33004847 0.5169535 -0.64807427 0.71064276 -0.6116849 0.4650579 0.5945314 -0.29525992 0.35196903 -0.74596804 0.28433082 -0.5798918 -0.5283981 -0.3814331 0.9196462 -0.7389609 0.48846203 -0.7602187 -0.4110869 0.6236794 -0.11718674 -0.46419492 0.60699517 0.5537438 0.32094273 -0.825315 -0.44702846 -0.5114494 0.46730548 0.13635007 -0.5564784 0.2817334 0.47415054 0.7806633 0.66634476 -0.47438824 0.8171659 0.74277943 0.74315476 -0.7544435 -0.6160638 0.3702301 -0.48115942 -0.29899907 -0.5685902 0.46495214 0.582189; -0.8036079 0.76664156 0.7992904 -0.5478456 -0.6917779 0.81435853 0.7884773 -0.7046415 0.91458654 0.53820115 0.6024371 0.5932802 -0.66152114 -0.30727604 0.25936818 -0.8413539 0.3844233 -0.3185413 0.5801563 0.9140667 0.044629913 -0.38425717 0.2777968 0.6243712 -0.9892429 0.5779891 -0.49670935 0.3870821 -0.51140195 -0.6156997 0.7416675 0.28098327 0.60519713 0.32522485 -0.118935354 0.6114781 -0.47303912 0.56969064 -0.36664987 -0.5320119 0.2960991 -0.14276673 0.79531014 -0.16634718 0.41044682 0.7603904 0.6143773 -0.7320292 0.72055244 -0.33398923 0.74663126 0.5154569 -0.85711545 -0.20497832 0.78253883 -0.46478114 -0.82577217 -0.5276512 0.6726149 0.13234264 0.11968044 -0.37483534 -0.07735666 0.61588854 -0.19764447 -0.8464609 -0.5605511 -0.49175823 0.74811906 -0.9206521 -0.8039684 -0.68959945 0.98410475 0.663525 -0.67753625 0.33364737 0.6781982 0.5386823 -0.5718905 -0.49339175; 0.9463456 -0.7223752 -0.84855026 0.033164512 0.89319146 -0.66165066 -0.7060642 0.6275402 -1.0145019 -0.25807634 0.0149327945 -0.56576 0.7079392 -0.33309355 -0.6643271 0.9355382 -0.100419 -0.37065485 -0.72748303 -0.81904405 0.32461652 0.046652928 0.029570749 -0.30387914 0.7260841 -0.8184996 -0.185716 -0.7137052 0.20818032 0.7513645 -0.5881545 -0.5451849 -0.19098644 -0.046679396 -0.5145769 -0.75634253 -0.14689125 -0.72550374 -0.25845316 -0.08813596 0.38395903 -0.4379335 -0.6456758 -0.42706928 0.12157967 -0.78683406 -0.21610878 0.686593 -0.6477113 0.19682021 -0.9170514 -0.59810966 0.66164446 0.59552175 -0.6879645 0.08934134 0.95717084 -0.26170135 -0.5544073 0.40420112 0.4587566 0.73230386 -0.47738424 -0.8594932 -0.22428763 0.92944884 1.1038104 0.56906825 -0.918422 0.66353846 1.0524297 0.104432985 -0.9793887 -0.6170559 0.060114376 -0.12268306 -0.45821166 -1.0497321 0.66443807 0.67853665; -0.31400374 0.4305817 0.3710929 0.6781595 -0.33767802 0.30347735 0.5361468 -0.3589043 0.33504608 -0.6025694 -0.69384515 0.44417793 -0.35511616 0.48415452 0.6079306 -0.47546452 -0.71564394 0.7053056 0.6111511 0.28505933 -0.6110548 0.7019005 -0.58763474 -0.9560376 -0.61209357 0.30769095 0.60732204 -0.89589566 0.5853002 -0.56505686 0.51973426 0.5845369 -0.6750932 -0.56523925 0.85184556 0.49641994 0.67992216 -0.8884185 0.7917727 0.8317553 -0.7999158 0.5345581 0.3214347 0.6316747 -0.615658 0.54348344 -0.6504297 -0.25035924 0.5575557 0.7608823 0.48347527 -0.85109985 -0.3571201 -0.73764884 0.34184256 0.6720033 -0.5241499 0.64732635 0.5146836 -0.5263377 -0.689862 -0.18106279 0.4382061 0.289097 0.6457539 -0.45155802 -0.52211016 0.24431266 0.39329237 -0.37030947 -0.34349418 0.7951158 0.2842098 0.3596981 0.7618338 -0.73452264 0.49540588 0.37632814 0.03938209 -0.25848567; -0.08356756 0.25068647 -0.32936105 0.91358805 -0.31649196 0.49832296 0.25245622 -0.42776638 0.17767134 -0.63435996 -0.95431715 0.33138525 0.31219327 0.7217854 0.7140267 -0.23653954 -0.8987279 0.77760816 0.52074367 0.35751587 -0.5753355 0.8821826 -0.62169003 -0.9311541 -0.20810446 0.109670654 0.7537199 -0.53292876 0.9350506 -0.15276314 0.34917316 0.5084466 -0.92047966 -0.9941854 0.6770081 0.3056041 0.9760262 -0.6971261 0.5598263 0.75136894 -0.67233646 0.65755564 0.36050245 0.8890996 -0.6417076 0.25121924 -0.83536696 -0.5733078 0.44018707 0.76224303 0.23727047 -0.5600499 -0.122189224 -0.63132 -0.38092855 0.8713118 0.12334928 0.69293195 0.39232734 -0.8155618 -0.64938164 0.32523212 0.7409564 0.36179674 0.5821901 -0.1509598 -0.22511645 0.38866118 -0.1760459 -0.3730804 -0.15975702 0.9070344 0.10433924 0.31087005 0.95668983 -0.8709715 0.48426437 -0.014914554 0.47303465 0.22470804; -0.86453557 0.98295534 0.76909333 -0.2581812 -0.73014927 0.7908697 0.70127857 -0.60573274 0.88925683 0.6676843 -0.21196838 0.77172035 -0.6465453 0.12728117 0.57838297 -0.58352244 0.19301859 0.4242965 0.8066082 0.94928646 -0.39434436 -0.14705685 -0.14997032 0.24205555 -0.92301327 0.9314395 -0.048953958 0.4728482 -0.32919616 -1.0232748 0.64210886 0.5448168 0.55933756 0.21037328 0.10937427 0.7095682 -0.20278955 0.37561315 0.23926985 0.09827665 -0.13272347 0.12974484 0.8527365 0.12782268 0.05243136 0.93223786 0.3666324 -0.9476652 0.68020445 -0.2711604 0.88598233 0.3150831 -0.7421914 -0.20230475 0.71285063 -0.2620132 -0.87032646 -0.08952162 0.805954 -0.48117614 -0.0075240433 -0.46595275 0.48379555 0.8769132 0.21270205 -0.8457802 -1.0133582 -0.7340771 0.75480634 -1.0103914 -1.0010093 -0.36680767 0.94555604 0.8057947 -0.4305935 0.17017888 0.5496006 0.74338484 -0.46338868 -0.55745286; 0.23118395 -0.44654053 0.46223596 -0.7972561 0.39362818 -0.566355 -0.102646105 0.29812163 -0.40231922 0.95066476 0.64315736 -0.47740066 -0.41051647 -0.8286725 -0.64255595 0.42432308 0.63518435 -0.7051117 -0.3878762 -0.5524308 0.84132 -0.59253645 0.7583447 0.9756146 0.2638783 -0.05495263 -0.66230243 0.5972967 -0.93464833 0.41489664 -0.36523858 -0.78076905 0.70824033 0.8948426 -0.7784773 -0.26458982 -0.6874622 0.6714911 -0.781441 -0.7224284 0.6403324 -0.8859585 -0.21208724 -0.7994145 0.87771827 -0.5307292 0.9328032 0.32238957 -0.1344084 -0.62701434 -0.04743467 0.44411424 0.4076689 0.6605583 0.34690237 -0.6274391 0.03449642 -0.845163 -0.55337757 0.866181 0.7582565 -0.13253713 -0.64769894 -0.14669698 -0.64384556 0.08031259 0.069650285 -0.8352336 -0.087034985 0.44253913 0.51804036 -0.8681287 -0.4720085 -0.16690232 -0.95558727 0.9509483 -0.27773824 -0.14581451 -0.41332164 -0.13744134; 0.2073991 0.21620646 -0.3172306 0.86385125 -0.0835698 0.4601628 0.15122534 -0.37795368 -0.037457295 -0.5962545 -0.8858346 0.5383507 0.38585913 0.6766142 0.50590277 -0.4897446 -0.73875856 0.84485 0.26770023 0.29334956 -0.8637549 0.8382014 -0.9855304 -0.81756455 -0.27017102 -0.2256078 0.9473482 -0.7946937 0.7498716 -0.07541802 0.46159294 0.5618097 -0.6725021 -0.8126394 0.68001413 0.20647486 0.91005486 -0.5956362 0.574506 0.66226965 -0.88948995 0.62196755 0.078083105 0.8560298 -0.7837485 0.2011103 -0.80728924 -0.415487 0.023907764 0.7455452 0.02031553 -0.6564651 -0.2758447 -0.54248667 -0.41023672 0.950448 0.48118868 0.6678797 0.2970436 -0.551431 -0.6100829 0.35610256 0.7988299 -0.038817007 0.75648695 0.11791756 -0.15035929 0.6413749 -0.1482543 -0.3328052 -0.031025458 0.91609466 0.087925576 0.13867949 0.8175306 -0.7238217 0.31437927 0.13948478 0.30797222 0.35588473; -0.8167405 0.8633277 1.0182542 0.1994808 -1.0010271 0.8107934 0.9781185 -0.5119655 0.78815866 0.033507746 -0.38999397 0.7308417 -0.83268756 0.53002 0.30938143 -0.6895711 -0.051891916 0.69431436 0.8226984 0.52948976 -0.40669253 0.39943516 -0.34238243 0.17448312 -0.48206043 0.6092284 0.12811251 0.69767237 0.040049173 -0.77554846 0.770823 0.5028714 0.29623866 -0.44443345 0.43343493 0.9473727 0.15068789 0.2766725 0.59920424 0.3603902 -0.63439703 0.3190146 0.96016175 0.114010036 -0.24023062 0.85095775 -0.24381647 -0.6382211 0.7799682 0.24149747 0.8665979 0.67522746 -0.8970999 -0.5305208 0.8397996 0.1310727 -0.6707967 0.16027097 0.495304 -0.2145161 -0.46383363 -1.01522 0.4149168 0.6125271 0.48437995 -0.86918646 -0.63898224 -0.4779567 0.74525464 -0.8246952 -0.6154232 0.2275346 0.7020598 0.9794944 0.18653679 -0.27415547 0.43375146 0.84719193 -0.98129135 -0.6775317; 0.67462564 -0.5914272 -0.5901282 0.5867969 0.72980183 -0.48920873 -0.6858564 -0.21054988 -0.32225493 -0.62991804 -0.5287608 -0.3043474 0.65937424 0.46775833 0.9060448 0.42405608 -0.90314674 0.6798618 -0.8185868 -0.61228126 -0.75081414 0.56043726 -0.626051 -0.76863337 0.30620548 -0.32119125 0.90198874 -0.5942357 0.6733926 0.3414164 -0.30137506 0.9255351 -0.6436619 -0.3637977 1.0144371 -0.7629672 0.89840907 -0.5015002 0.4708326 0.8636604 -0.42355138 0.76769066 -0.7196667 1.0161718 -0.39157286 -0.8135747 -0.46879774 0.6787948 -0.7607217 0.6819538 -0.2157263 -0.6860336 0.7687318 -1.1499625 -0.489841 0.5726094 0.58094144 0.8776142 0.5525787 -0.8957526 -0.78453916 0.79747725 0.8003523 -0.5728772 0.90881133 0.31993172 0.5636584 0.29669645 -0.5797961 0.49463147 0.3156737 0.46517748 -0.4194719 -0.58847755 0.90478367 -0.59603584 0.572253 -0.48180538 0.6079689 0.6344739; 0.071724206 0.088906415 -0.55646276 0.7704257 -0.3002049 0.20575467 -0.116210334 -0.31159347 -0.09219774 -0.5333049 -0.7078772 0.25385433 0.3771218 0.7393395 0.8329547 -0.39216468 -0.74813944 0.69358087 0.27914825 0.3888224 -0.6747312 0.6671681 -0.73544425 -0.64297634 -0.20972142 -0.18790163 1.0003954 -0.43146366 0.7725391 0.0077699893 0.22105439 0.6995126 -0.77247715 -1.0312299 0.8524248 0.054428317 0.64423287 -0.78259987 0.9579165 0.69624454 -0.668584 1.004833 -0.031186834 0.9341358 -0.8051467 0.19465396 -0.7562795 -0.3976218 -0.0934909 1.0024669 -0.054945357 -0.53610766 -0.1174696 -0.70549595 -0.60221094 0.7083162 0.36021334 0.9528258 0.39821628 -0.65370035 -0.6306891 0.53194857 0.58382714 -0.12762775 0.6735038 0.13248725 -0.045141872 0.69821537 -0.30500174 0.106686525 -0.3024011 0.8979239 -0.12824233 -0.10636666 0.7493677 -0.98013836 0.42816252 0.07464679 0.70147264 0.46108145; 0.13358185 -0.00993287 -0.35780254 0.80490035 -0.020046175 0.24292909 0.007448801 -0.20396252 -0.2054027 -0.58709264 -0.9148848 0.37173018 0.47651634 0.74970335 0.8419168 -0.3507905 -0.9347393 0.6673021 0.14787231 0.15266119 -0.8135492 0.7792544 -0.7678244 -0.91242754 -0.35447508 -0.06654083 0.91001177 -0.6081724 0.7636977 -0.11667863 0.18850869 0.90128976 -0.62240255 -1.0220634 0.7774392 -0.30261457 0.7895823 -0.48543054 0.71884304 0.7429817 -0.8592336 0.8231822 -0.0513855 0.9613681 -0.9769212 0.22322394 -0.62434006 -0.25738308 0.022224013 0.84246296 -0.19587961 -0.4776633 0.074170865 -0.59716195 -0.3926755 0.77373433 0.5603546 0.65196687 0.638518 -0.6361705 -0.8609037 0.5683014 0.82293415 -0.2359172 0.9810779 0.3299566 0.061455745 0.4582074 -0.2284231 -0.13168861 -0.10274632 0.7805261 -0.040445685 -0.18527023 0.7556924 -0.72872216 0.33355114 -0.28013694 0.40346953 0.4460827; -0.6600918 0.772782 0.89625067 0.32719386 -0.81456757 0.7173408 0.8108354 -0.50014216 1.0429386 0.35299042 -0.1519625 0.74118525 -0.63549817 0.3409754 0.27225175 -0.8822215 -0.16083959 0.32311633 0.75689256 0.6928899 -0.15050593 0.059339184 -0.3418931 0.24993749 -0.6927896 0.9477706 0.120019846 0.34041086 0.090475805 -0.9377535 0.5477889 0.63592744 0.28648555 0.0043523638 0.6313268 0.80658585 -0.10000556 0.43999067 0.3914339 0.5434679 -0.4637615 0.3611057 0.97762084 0.57293177 -0.31168607 0.59826034 0.13980098 -0.6095293 0.8387351 0.04981923 1.0319656 0.4997787 -0.62896657 -0.35765332 0.7151248 -0.18064669 -1.0597855 0.086658634 0.8996926 -0.3116911 -0.62066954 -0.5552845 0.42686597 0.71340704 0.47720084 -0.8590703 -0.9764754 -0.80126715 0.95621943 -0.81130296 -0.6571552 0.21604536 0.6699685 1.0322913 0.17151737 0.014643361 0.41020235 0.739768 -0.7326234 -0.8403505; -0.15435101 -0.4034486 0.30610833 -0.7464967 0.3764299 -0.33790705 -0.0014493039 0.49246714 0.07069468 0.80946505 0.83862084 -0.3189473 -0.4664499 -0.74923545 -0.5556918 0.45563114 0.7435471 -0.859474 -0.07469085 -0.48153967 0.6654985 -0.8001589 0.80972546 0.70576453 0.20543866 -0.037117712 -0.7931514 0.60550106 -0.764544 0.30548254 -0.41199502 -0.5976546 0.6104589 0.8246211 -0.96361136 -0.05267668 -0.8326803 0.6169723 -0.8991504 -1.0417285 0.8831995 -0.7177807 -0.18113835 -0.82635564 0.8000111 -0.12677613 0.93557906 0.17071183 -0.24286601 -0.9678587 0.2591973 0.39724883 -0.06252305 0.87024695 0.40930107 -0.9345082 -0.27377614 -0.9394729 -0.5668488 0.7253626 0.7329821 -0.47724336 -0.81406254 0.0053326692 -0.754666 0.062778115 -0.22146091 -0.50712997 0.21073632 0.21207103 0.06748136 -0.94012374 -0.33919656 -0.114239134 -0.6950721 0.9430432 -0.30104846 -0.08055567 -0.33431128 -0.53965443; 0.63984215 -0.49902537 -0.47315392 -0.82115006 0.5890641 -0.33385023 -0.596982 0.26489502 -0.3781224 0.6856328 0.692167 -0.671523 0.43326744 -0.6395111 -0.3579941 0.5285258 0.43477884 -0.6585809 -0.40628228 -0.2569516 0.7373568 -0.68999004 0.65492064 0.44457102 0.46147057 -0.5594479 -0.52124244 0.72022086 -0.7435265 0.37911615 -0.62364876 -0.6962611 0.47077653 0.6257547 -0.5784841 -0.43233672 -0.545102 0.81020373 -0.78864795 -0.66050065 0.5494879 -0.60603654 -0.666469 -0.50458586 0.89593065 -0.3671729 0.919806 0.6532105 -0.4121116 -0.5052548 -0.22589147 0.20330115 0.38065043 0.45467472 -0.27382383 -0.54884654 0.119232155 -0.5196585 -0.24410571 0.6420077 0.49993858 0.4522242 -0.9084234 -0.40237 -0.7284901 0.26024577 0.34006283 -0.5821087 -0.4028614 0.61674994 0.45029446 -0.93634754 -0.24522273 -0.5602707 -0.50431824 0.7634032 -0.27356678 -0.55742896 0.5092655 0.5328362; 1.0114776 -0.8514356 -0.9671909 -0.40791437 0.6915648 -0.8004603 -0.88793975 0.50724983 -0.92161554 -0.023283027 0.21566845 -0.86829656 0.93250877 -0.3325615 -0.4497931 0.6670963 0.31481823 -0.5281696 -0.81799203 -0.8676147 0.3762417 -0.3108407 0.5102973 -0.1975166 0.46543807 -0.6083945 -0.1909797 -0.34118778 -0.17693193 0.7277606 -0.5139756 -0.2805519 -0.30926284 0.2776916 -0.2845411 -0.8889604 -0.023288654 -0.50333625 -0.5043375 -0.48007104 0.37014362 -0.33616602 -0.7997878 -0.41757628 0.5154314 -0.833952 0.18043257 0.61308026 -0.9530101 -0.3788221 -0.58547217 -0.2673673 0.86683846 0.35365152 -0.99359274 -0.09785234 1.0039401 -0.34600556 -0.44779563 0.43231383 0.4841377 0.6391445 -0.25912738 -0.9202655 -0.46844408 0.9237293 0.65735286 0.78821415 -0.7471487 0.77713394 0.61963964 -0.14057536 -0.7318457 -0.91416395 -0.20715283 0.2069268 -0.5468635 -0.606864 0.6791757 0.67131186; 0.85149634 -0.86040634 -0.5410756 0.19669574 0.656114 -0.9500288 -0.88538814 0.5422854 -0.94765145 -0.4640081 -0.19195609 -0.64735264 0.4939231 0.09291637 -0.56648767 0.86324936 -0.24554998 -0.01005132 -0.65544075 -0.942984 -0.050244406 0.20314132 -0.29537347 -0.3280658 0.86665684 -0.63291746 0.31623012 -0.4432935 0.6009629 0.90498 -0.7186476 -0.34272233 -0.4321156 -0.43258193 0.05558549 -0.8939072 0.27735853 -0.53848237 -0.18081047 0.09600071 0.108344555 -0.062871516 -0.7872238 0.026731405 -0.45848045 -0.7645587 -0.46757635 0.6964161 -0.60201967 0.30294004 -0.7209322 -0.4141981 0.6552485 0.08675466 -0.71814287 0.36336368 0.7128171 0.27867404 -0.55270296 0.16238458 -0.13226546 0.4783351 -0.3504874 -0.647479 0.012694088 0.75642896 0.7172198 0.61738235 -0.6189792 0.72155774 0.9329303 0.48127005 -0.75684345 -0.72664803 0.27273902 -0.33177394 -0.32796702 -0.7942235 0.4357933 0.57811; -0.7925899 0.7927796 0.63056266 0.16980442 -0.7027303 0.5018388 0.52953213 -0.5269502 0.7458217 -0.042826563 -0.37809712 0.63915944 -0.6451228 0.19306488 0.33196208 -0.55411 -0.33836487 0.29542133 0.7089121 0.57173866 -0.61471856 0.4118784 -0.4536502 -0.23566109 -0.8491335 0.8320553 0.5591085 0.13782354 0.192186 -0.8259475 0.46575716 0.57453436 -0.3770037 -0.44120526 0.51551664 0.7830121 0.51828194 0.36705506 0.19648921 0.39103422 -0.5345014 0.3565331 0.71361125 0.47946274 -0.32898384 0.74370307 -0.1269718 -0.63012254 0.5913115 0.20746486 0.7225467 0.20912595 -0.7315975 -0.6207136 0.9673923 0.525453 -0.9287847 0.60126555 0.65511304 -0.40847635 -0.5723889 -0.8633506 0.5419534 0.6368719 0.54761696 -0.7490057 -0.9220472 -0.93919134 0.7035028 -0.7056862 -0.7257137 0.25150082 0.8564546 0.8496149 0.498043 -0.41263822 0.2924998 0.65796006 -0.68341166 -0.83465505; 0.71184146 -0.61793333 -0.5243012 -0.33583173 0.6015418 -0.4194757 -0.8036669 0.39440438 -0.8509831 0.37708536 0.45811883 -0.48909992 0.636025 -0.56770366 -0.5899665 0.5813214 0.3630433 -0.5988597 -0.4463542 -0.33589226 0.4311532 -0.4800419 0.39385125 0.4033686 0.5054614 -0.6724958 -0.35873285 0.46498522 -0.5524359 0.5300798 -0.59604114 -0.63001823 0.45641494 0.34939343 -0.58405054 -0.6320086 -0.43665975 0.27255052 -0.24677902 -0.52785134 0.35855895 -0.5696147 -0.67185074 -0.4868887 0.24415366 -0.3315257 0.4643923 0.3140118 -0.45420885 -0.3493304 -0.7474607 -0.4796157 0.5219393 0.30745465 -0.7401179 -0.49868894 0.69282645 -0.6355188 -0.43299246 0.58608794 0.5161005 0.45801297 -0.5587794 -0.8031674 -0.49164563 0.67950547 0.5890124 0.036819343 -0.5443082 0.5618409 0.8458446 -0.52715117 -0.5445356 -0.6518951 -0.41575035 0.6096243 -0.53658825 -0.738371 0.44081476 0.75124806; 0.39129135 -0.05899293 -0.37847242 0.97745556 -0.15995088 0.5212848 -0.03125535 -0.5471634 -0.21137337 -0.79913956 -0.658942 0.6292986 0.59056073 0.93242455 0.85202634 -0.34543553 -0.69709533 1.0193645 0.19541405 0.28009135 -1.0202806 0.8616527 -0.87499815 -0.594 -0.3842475 -0.4138006 0.8540118 -0.7206358 0.83771485 0.20731555 0.30764118 0.8148364 -0.55951536 -0.98820597 0.9660242 -0.20962931 0.5541332 -0.56936044 0.68203217 0.7827488 -1.0151461 0.8532325 -0.008905882 0.7362406 -0.902429 0.047554627 -1.0230671 -0.33869907 0.022836424 0.9464342 -0.34421018 -0.4862212 0.0058049075 -0.8369924 -0.63355625 0.5824318 0.41596487 0.6013119 0.5408795 -0.75929785 -0.79501927 0.6379506 0.70009583 -0.13407703 0.7872441 0.32029063 0.09698375 0.55780965 -0.3468496 -0.05841412 -0.12586361 0.942894 -0.31940496 -0.07550018 0.72267973 -0.6925804 0.54508585 -0.14031047 0.6566234 0.3864887; 0.7959594 -0.85487986 -0.5207178 -0.14047545 0.9860329 -0.54378396 -0.7644199 0.49198863 -0.85048413 -0.24423184 0.3424247 -0.6805607 0.78527784 -0.43474093 -0.3158752 0.4723681 0.036749244 -0.68147707 -0.79904205 -0.5898903 0.536022 -0.2815787 0.5694121 -0.19219927 0.46541488 -0.65614736 0.033785325 -0.64002305 -0.082082264 0.51023096 -0.77923626 -0.23835677 -0.42758125 0.2783474 -0.15563445 -1.0336562 0.23805113 -0.46135208 -0.39765593 -0.1362674 0.4869257 -0.26316944 -0.59246343 -0.23297386 0.48668665 -1.0310246 0.26259002 0.64023364 -0.7529499 -0.058842096 -0.691784 -0.94505894 0.7610424 0.15323448 -0.65217453 0.10256348 0.72012186 0.03728614 -0.4932108 0.22381635 0.3993018 0.68856597 -0.4984811 -0.5522842 -0.33967522 0.750477 0.65884036 0.50724596 -0.88347286 0.509195 0.659412 -0.29037902 -0.7636493 -0.6591434 0.08487606 -0.043139208 -0.23477517 -0.9112014 0.92773443 0.8539356; 0.7475261 -0.64462554 -0.6650235 0.48992595 0.5732821 -0.67319685 -0.50550485 0.89898556 -0.6040461 -0.7161162 -0.51262546 -0.64252084 0.74396133 0.2932988 -0.17247401 1.0399327 -0.42344853 0.48286414 -0.5330735 -0.7765725 -0.38880745 0.51158005 -0.6451713 -0.3370237 0.96754897 -0.71667737 0.5991583 -0.4931247 0.51672727 0.6341996 -0.78406227 0.24597856 -0.4193758 -0.63949776 0.19547024 -0.43044418 0.6891619 -0.45291832 0.37537432 0.39395583 -0.21461236 0.5697439 -0.536798 0.1927347 -0.6411608 -0.7080336 -0.59848726 0.6286178 -0.5410204 0.69492894 -0.6946321 -0.5677649 0.8548367 -0.016207939 -0.41142723 0.6057637 0.788355 0.2982024 -0.56917095 -0.2528672 -0.46388385 0.66208285 0.13664635 -0.7056829 0.27523753 0.7038825 0.8728726 0.5617236 -0.6081862 0.58125067 0.73363656 0.56749505 -0.89740115 -0.76275903 0.5899143 -0.5903362 -0.3098926 -0.7474735 0.69284785 0.5262483; 0.9511495 -0.56375915 -0.9124551 -0.5795384 1.0164537 -0.7486328 -0.9309125 0.42865112 -0.52171767 -0.0777026 0.54622066 -0.7733525 0.7553088 -0.6566086 -0.50950193 0.648203 0.067945726 -0.4330738 -0.9751138 -0.5067957 0.2916615 -0.33168876 0.4640982 -0.13810742 0.50866884 -0.78683674 0.019351894 -0.5672935 0.11899042 0.5031433 -0.63698196 -0.4237527 -0.43130162 0.5437907 -0.24939033 -1.0651013 -0.045465298 -0.5534298 -0.6263764 -0.43879965 0.58145297 -0.36953676 -0.78396034 -0.10265534 0.56249267 -0.6884608 0.46574926 0.68434584 -0.9245294 -0.13348436 -0.7257432 -0.75257504 0.7665872 0.49515143 -0.9579503 0.0053100386 0.80719876 -0.21043867 -0.63086605 0.40596825 0.29658931 1.0792273 -0.6823458 -0.5497281 -0.11296605 0.72899115 0.858454 0.7494137 -0.61705893 0.8419234 0.7103841 -0.16258986 -0.8148307 -1.0068591 -0.24320586 0.41990808 -0.34826425 -0.7179637 0.7342272 0.957842; 0.5593277 -0.06347121 -0.65813977 0.917863 0.10542155 0.2832114 -0.3809892 -0.6209215 -0.44179103 -0.61639863 -0.7369268 0.25180086 0.39626735 0.95690775 0.59727573 -0.18680519 -0.78312874 0.68872255 -0.042682335 0.256523 -0.69567037 0.71210927 -0.8313129 -0.77987236 -0.19089861 -0.32542244 0.92518264 -0.7043685 0.6196177 0.14603099 0.3023873 0.66431946 -0.5483303 -0.7447663 1.0150146 -0.4139745 0.8575969 -0.42876562 0.91372687 0.87124336 -0.9427988 0.74673873 -0.3356763 0.91719955 -0.7824297 -0.37479025 -0.76126665 -0.04237599 -0.1956712 0.7907766 -0.51674175 -0.54609615 0.46084395 -0.7077973 -0.6547703 0.66734606 0.27856398 0.8106799 0.33639908 -0.72517896 -0.9625961 0.5760889 0.63403755 -0.22485171 0.8659075 0.46743983 0.3911118 0.56214684 -0.32660452 0.35762277 -0.083439164 0.82789785 -0.21175581 -0.4484747 0.752509 -0.8710955 0.6986832 -0.46714482 0.5581655 0.7405837; -0.21242279 -0.1462722 0.2581954 -0.7478501 0.006325955 -0.37494257 0.13433512 0.60045904 0.14101553 0.6662216 0.7216954 -0.5377694 -0.48597974 -0.93210965 -0.61326444 0.6258673 1.0308485 -0.7510651 -0.29398003 -0.4071796 0.75522965 -0.7098003 0.7469642 0.64480674 0.29718718 0.34571669 -0.80278116 0.5423764 -0.8352453 -0.09304578 -0.37431225 -0.8571997 0.6645584 0.97962755 -0.75176316 -0.032947425 -1.0084524 0.50194186 -0.8678948 -0.7356789 0.8863575 -0.961237 -0.25385264 -0.7286287 0.8266846 -0.1667123 0.73133004 0.36326906 0.03972831 -0.9021192 0.28053847 0.7258299 -0.03187712 0.6445121 0.20535217 -0.75167924 -0.16679583 -0.8049542 -0.35041 0.89344066 0.72304046 -0.6704006 -0.7060844 0.0385168 -0.9212702 -0.07558827 -0.24826291 -0.5834289 0.07180928 0.2609647 0.044348698 -0.85608613 -0.24974068 -0.028456543 -0.7986789 0.7166989 -0.63156974 0.3232456 -0.46545875 -0.48451382; -0.6508319 0.95991075 0.5523548 0.060623307 -0.7708937 0.593472 0.65251815 -0.9346106 1.0547729 0.40861124 -0.29981902 0.6504321 -0.82470846 0.29413328 0.31825107 -0.8498979 0.2702245 0.24174199 0.79054475 0.948268 -0.40699217 0.2618267 0.10982736 0.39336532 -0.70772386 0.9066758 -0.1980385 0.32346293 -0.038111966 -0.88783276 0.611894 0.38091466 0.2701094 0.06385213 0.29669547 0.6362879 -0.15746906 0.7811418 0.42482942 0.23264347 -0.038690362 0.39094648 0.68390095 0.45848882 0.1490416 0.55440694 0.051561415 -0.71496725 0.68061316 -0.15004592 0.7758766 0.29307634 -0.7119277 -0.5830557 0.7238793 0.02785562 -1.0953872 0.040116623 0.7563117 -0.49955204 -0.16394912 -0.7226842 0.10512112 0.9593676 0.41721 -1.0492343 -1.1075009 -0.73369145 0.9973127 -0.7803654 -1.0336114 -0.3073105 1.0685084 0.86565286 -0.15921207 0.0118873445 0.56459004 0.8105981 -0.57743084 -0.67623; 0.5664196 -0.5067877 -0.12828308 -0.8358634 0.46380198 -0.50940925 -0.43044028 0.4353265 -0.27774677 0.60611415 0.48706943 -0.2855008 0.35959983 -0.7109096 -0.37072057 0.5345695 0.742748 -0.58182764 -0.3783246 -0.45293877 0.7031666 -0.79670364 0.8037644 0.6357337 0.42459998 -0.516385 -0.6780983 1.0606315 -0.7245683 0.49096826 -0.44435674 -0.76178575 0.7848988 0.7596044 -0.73920023 -0.47097296 -0.5718031 0.975625 -0.50731194 -0.58070356 0.5723391 -0.7372648 -0.37376562 -0.55156887 0.46535262 -0.45540187 0.74310195 0.40418535 -0.31563914 -0.580852 -0.51364994 0.36553347 0.39354727 0.61206317 -0.410106 -0.6910967 0.5751147 -0.79666203 -0.47115058 0.57747626 0.5593616 0.31867152 -0.54681927 -0.47562355 -0.51529014 0.31856 0.40411362 -0.38960114 -0.25850046 0.49124467 0.5504125 -0.4612069 -0.42867097 -0.51304394 -0.70279944 0.77789944 -0.3527773 -0.34791604 0.2062421 0.124022275; -1.0994841 0.7027967 0.71580845 0.13453004 -0.78218687 0.678765 0.7789672 -0.41250768 0.6860836 0.55622 -0.32432646 0.7768841 -0.6343217 0.4111676 0.5385153 -0.5263232 0.24205597 0.44764882 0.8243276 0.56460804 -0.30460262 0.37306312 -0.29811177 0.2824311 -0.6106961 0.68875486 -0.25212342 0.7111988 -0.0042743916 -0.7378902 0.8807121 0.4529781 0.8272426 -0.45853314 0.25728506 0.71170264 -0.19795911 0.57015866 0.4038285 0.070465356 -0.40060717 -0.16956437 0.58925825 0.37108257 -0.30184686 0.78465533 -0.38638213 -0.67871386 0.91931766 0.10596561 0.82541686 1.4020766 -0.6719228 -0.14271466 0.5536113 -0.045539424 -0.38430214 -0.2418265 0.461029 -0.22224072 -0.21544622 -0.87060666 0.590332 0.59924775 0.28124642 -0.7496785 -0.6304976 -0.18717334 0.55293274 -0.7008086 -0.6547574 0.30849427 0.58578354 0.96592754 -0.20813425 0.08830402 0.26999977 0.704241 -1.1708272 -0.9253619; 0.032033995 -0.23130207 0.29633516 -0.94401133 0.3472803 -0.53857577 -0.23406334 0.21202117 -0.14481266 0.9360933 0.8881505 -0.5001395 -0.5416392 -0.6090332 -0.7576122 0.4039583 0.6970242 -0.8542872 -0.09732229 -0.1972808 0.8542032 -0.76904863 0.8755753 0.68110925 0.31106457 -0.02902219 -0.92316705 0.48702 -0.91356397 0.06093066 -0.2226835 -0.66237605 0.768268 0.81678003 -0.8293173 -0.20298448 -0.7109203 0.76188475 -0.5310178 -1.0197934 0.8025978 -0.6681409 -0.35870454 -0.7547611 0.7885006 -0.31886053 1.0571582 0.25524285 -0.36155173 -0.81401455 -0.16434766 0.47815502 0.13355558 0.7913645 0.22055978 -0.86023086 -0.28857598 -0.519364 -0.57862055 0.92874736 0.6469641 -0.28717208 -0.67189324 -0.017070334 -0.87804765 -0.027147453 0.24381886 -0.40047646 0.14799435 0.2912258 0.20241146 -1.0551453 -0.14306961 -0.16169903 -0.70940816 0.9921241 -0.5780556 0.02700356 -0.47673014 -0.55269045; 0.70630914 -0.65010595 -0.4283715 0.6162818 0.522113 -0.9348025 -0.48811194 0.8767066 -0.8154212 -0.44362202 -0.5027716 -0.82895404 0.7391883 0.24740948 -0.19478011 0.8809234 -0.6080719 0.3478489 -0.6204482 -0.9754951 -0.4397212 0.41819984 -0.63972795 -0.7093081 0.7878106 -0.49282262 0.3486889 -0.6513327 0.64262754 0.85898006 -0.9291794 -0.043952655 -0.5491323 -0.6639361 0.26846796 -0.6864454 0.6984119 -0.71652204 0.42154425 0.5189607 -0.24207786 0.51989913 -0.8890284 0.51356107 -0.617194 -0.8914183 -0.7363173 0.83866984 -0.7011617 0.37982485 -0.79977757 -0.59181315 0.5541208 -0.044980526 -0.6456028 0.4277435 0.58631086 0.39188385 -0.9257572 -0.4332445 -0.2518119 0.4066828 0.069606505 -0.67955965 0.5016939 0.5433599 0.50885296 0.7962652 -0.8025885 0.5922116 0.7874738 0.51113224 -0.7339781 -0.5423273 0.7367082 -0.6876821 -0.6686571 -0.6541206 0.56744665 0.69885075; -0.2348898 0.30475962 -0.44865564 0.692746 -0.4902773 0.54583144 0.35847795 -0.47055733 0.341946 -0.9002528 -0.70032066 0.31991154 0.32678893 0.82802725 0.5931188 -0.3083072 -0.8045804 0.7979938 0.4781951 0.45561516 -0.84436244 0.77611107 -0.62802774 -0.75568104 -0.5459635 0.008960646 0.99171793 -0.60877067 0.89443856 -0.4956994 0.46795097 0.5447786 -0.8465909 -0.9135638 0.5762678 0.10905505 0.70787764 -0.83464134 0.75575054 0.8057039 -0.6974888 0.5612686 0.38797417 0.84895855 -0.6950861 0.31778288 -0.87106603 -0.5705377 0.21265873 0.8882927 0.22310828 -0.48696738 -0.11492998 -0.5366991 -0.3428418 0.81571454 -0.095269196 0.55178005 0.28314593 -0.5567485 -0.91599756 0.2807492 0.7023698 0.1193073 0.8040973 -0.25238138 -0.030525971 0.5213367 0.10933795 -0.15851727 -0.33424255 0.82154936 0.40291592 0.16300914 0.9569102 -0.92285377 0.5972206 0.18590535 0.45922458 0.49773088; 0.10435719 -0.29993978 0.5373337 -0.7726274 0.2956359 -0.5776967 -0.22229682 0.47815886 -0.22439764 0.7443517 0.88192296 -0.41491088 -0.57530415 -0.81655663 -0.47555307 0.2910485 0.7483516 -0.8056758 -0.35607868 -0.22795242 0.8091967 -0.976913 0.79839504 0.81138617 0.32029298 0.11837194 -0.7018809 0.68376166 -0.67479473 0.19691823 -0.45617422 -0.70479196 0.52623147 0.9496699 -0.90043956 -0.23282263 -0.95618176 0.74599904 -0.72199833 -1.0157442 0.60112685 -0.8218945 -0.38462213 -0.94569814 0.71079457 -0.38660786 0.8173425 0.20090462 -0.30545238 -0.914835 0.05152795 0.29179785 0.23818187 0.65884763 0.4436815 -0.81728405 -0.3507475 -0.7350694 -0.5465351 0.6328113 0.6880962 -0.3299556 -0.6545937 -0.21974246 -0.7319683 -0.103188105 0.0072429217 -0.5639114 0.30554494 0.31829304 0.16899806 -0.9508168 -0.055818092 -0.18760985 -0.6447502 1.0259162 -0.60896605 -0.03594519 -0.5924793 -0.34414327; -0.48789486 0.70460755 0.74648774 -0.66362625 -0.74486655 0.7437108 0.5637276 -0.63817656 0.6519757 0.75717765 0.23663764 0.5137117 -0.7442265 -0.13860486 0.20914936 -0.9371482 0.62812316 -0.06652252 0.6952395 0.8030498 0.10252612 -0.4234396 0.58028704 0.6562351 -0.8617345 0.82229865 -0.32255533 0.74870837 -0.34321678 -0.8472238 0.98008925 0.36360937 0.64331317 0.4816498 -0.17379124 0.7584145 -0.4756986 0.7808798 -0.09156229 -0.25307575 0.19935857 -0.026412142 0.5884318 -0.05973174 0.42432934 0.8263175 0.449028 -0.65922195 0.83190614 -0.57144797 0.8666383 0.5840866 -0.60319835 -0.3845826 0.55077064 -0.51360345 -0.6530432 -0.107710764 0.84189045 -0.04250525 0.25421178 -0.7224676 0.028613737 0.67078984 -0.26574656 -0.80631995 -0.5711212 -0.7969261 0.8509209 -0.9317696 -0.7001366 -0.59222555 0.75240326 0.85403067 -0.3953703 0.2712011 0.74385464 0.70044255 -0.38192323 -0.7460859; 0.540518 -0.22830701 -0.68253934 0.623638 0.25793114 0.104589164 -0.4311448 -0.18171872 -0.32586595 -0.7949468 -0.8819405 0.1685245 0.44470692 0.742292 0.75703907 -0.15778027 -0.5492111 0.8900956 -0.35824898 -0.055006117 -0.6438625 0.85973555 -0.64023894 -0.5668897 0.13531826 -0.4496242 0.8075297 -0.76937 0.7491625 0.29286492 0.033930108 0.7280299 -0.53126156 -0.695387 0.8506954 -0.38796738 0.57129925 -0.5320775 0.7610696 0.83269674 -0.7077516 0.72305155 -0.2219674 0.968521 -0.65371394 -0.57493776 -0.48851493 0.13635635 -0.34056672 0.6586312 -0.5335871 -0.5453544 0.43317986 -0.74842125 -0.52906954 0.623499 0.6078262 0.5972457 0.29063743 -0.6809989 -0.6976148 0.56679577 1.0090752 -0.39764223 0.85309476 0.37423855 0.47558045 0.7278483 -0.53558725 0.23050348 0.3884733 0.5668972 -0.42318904 -0.5938498 0.8083729 -0.72716963 0.57520676 -0.50956225 0.5343637 0.55660343; -0.7783528 0.85779625 0.7766646 -0.39537847 -0.87956244 0.87381476 0.80032986 -0.81597036 0.4796409 0.5052452 0.3675202 1.070989 -0.690381 -0.30790955 0.23048218 -0.8668406 0.5057694 -0.039287414 0.9482689 0.81575084 0.0988912 -0.24935482 0.38085127 0.6215717 -0.73503816 0.7708757 -0.5392362 0.7434906 -0.6566698 -0.7666758 0.75514185 -0.2231951 0.5467016 0.51421756 -0.22392362 0.89907426 -0.55615604 0.48091936 -0.2978689 -0.24816026 0.23326601 -0.6000167 0.75591457 -0.41842481 0.53686136 1.0035908 0.5028894 -1.0248363 0.905078 -0.597233 0.54220146 0.729651 -0.7724775 0.31902722 0.7395911 -0.6357299 -0.49710655 -0.562479 0.77258587 0.17295693 0.37122712 -0.5377766 0.11962055 0.40606344 -0.44178993 -0.81485194 -0.50133497 -0.65255 0.36446714 -0.8944873 -0.5259667 -0.2757228 0.690569 0.5453645 -0.40601206 0.6063261 0.52950406 0.73792166 -0.7183355 -0.5696452; -0.16335072 -0.099426076 0.5265241 -0.7768237 0.10783714 -0.4290263 -0.08906538 0.29246292 0.13187198 0.75816816 1.0297061 -0.35838348 -0.4760621 -0.77890503 -0.51912075 0.45946628 0.8572433 -0.89875054 0.02650084 -0.3100249 0.65508425 -0.76325566 1.0109452 0.55342406 0.3042365 0.22291999 -0.7853664 0.6135783 -0.74668735 -0.04095576 -0.29750744 -0.82916623 0.5493815 0.68688554 -0.81008023 -0.033660688 -0.8210836 0.73046184 -0.8780968 -0.64816695 0.89088434 -0.6458017 0.07239269 -0.821352 0.72755575 -0.10643859 0.9664241 0.18208116 0.23752426 -0.95470154 0.40573835 0.5771099 0.05555075 0.5966497 0.4851711 -0.6740047 -0.49890432 -0.7049067 -0.28225544 0.72479516 0.6629403 -0.47748208 -0.66927826 0.022298913 -0.6543012 -0.45082313 -0.25245368 -0.7759141 0.373255 -0.14335494 0.29023153 -0.8861285 -0.18650505 0.22310139 -0.67090106 0.82586205 -0.38261235 0.25006676 -0.6041666 -0.69201756; 0.30165416 -0.37319854 0.14158723 -0.6132097 0.36093113 -0.6080918 -0.41669136 0.6296644 -0.4570031 0.9087418 0.733201 -0.49392706 -0.07644342 -0.7271745 -0.77843153 0.30541965 0.7871986 -0.84760064 -0.25752756 -0.27813464 0.6861931 -0.69638646 0.550147 0.7257553 0.28410432 -0.5014527 -0.7087344 0.70028204 -0.69995385 0.23878059 -0.31713995 -0.6357927 1.0067724 0.87412286 -0.62985903 -0.26038957 -0.92677927 0.92082024 -0.62881374 -0.82025486 0.4913323 -0.7855262 -0.4408626 -0.63395673 0.86903465 -0.32637906 0.86718726 0.57513404 -0.31144717 -0.641741 -0.45860744 0.7737486 0.19062707 0.60525733 -0.16612843 -0.7486487 0.13047113 -0.79247797 -0.4470608 0.7620841 0.5860632 -0.24320105 -0.7773026 -0.3671825 -0.8479533 0.30224836 0.4283385 -0.34790388 -0.4402082 0.44397184 0.36613166 -0.9244026 -0.40263915 -0.379481 -0.65477705 0.67562866 -0.38704813 -0.4666101 -0.3384611 0.046537593; -0.36590093 -0.056809664 0.71502197 -0.73534507 -0.0648169 -0.4321501 0.31000566 0.28271046 0.25591296 0.5234196 0.67853814 -0.4916717 -0.6471024 -0.71496576 -0.7520661 0.33299994 0.9659328 -0.7054157 -0.050770737 -0.13335405 0.8815608 -0.73940295 0.87138534 0.56952757 0.18281487 0.19096963 -0.6678094 0.44627962 -0.7548804 0.012917675 -0.20902114 -0.7662226 0.53275293 0.9161994 -0.6903199 0.015411154 -0.6804812 0.66580945 -0.9572378 -0.7182707 0.5848537 -0.869308 -0.17362659 -0.7087042 0.6449765 0.10061632 0.8292758 0.1359347 0.12144698 -0.63915646 0.10990655 0.58000374 0.10911657 0.9018111 0.334832 -0.85479444 -0.43818903 -1.133445 -0.56673956 0.64392126 0.958269 -0.70725995 -0.7615096 -0.06579439 -0.77940387 -0.3555646 -0.12914163 -0.57977164 0.37286302 -0.18481323 0.30099812 -0.807292 -0.003102054 0.25417316 -0.66446334 0.99239296 -0.6463343 0.0070413235 -0.58460605 -0.7228561; -0.34712097 0.25688577 0.095746964 0.74654084 -0.71652836 0.477809 0.6670158 -0.26412296 0.023002317 -0.9608583 -0.9549351 0.65873057 0.32737815 0.78272015 0.48818624 -0.2696087 -0.6447182 0.8750356 0.69848746 0.12818563 -0.5005892 0.65087676 -0.8858273 -0.71536165 -0.039549146 0.24559067 0.68338263 -0.6874529 0.8314963 -0.27588752 0.6631529 0.29466146 -0.42512986 -1.0401301 0.69719434 0.63915306 0.39901686 -0.85680246 0.91913676 0.6782683 -0.9064129 0.5210979 0.34393087 0.35332775 -1.1007378 0.7652144 -1.0674118 -0.36536753 0.6164297 0.88704205 0.23778427 -0.046518262 -0.5019802 -0.52287287 -0.11073354 0.43079394 0.23818962 0.53660434 0.21563026 -0.7522625 -0.52218074 -0.06455336 0.96160483 0.019391002 0.4184895 -0.04891119 0.0673049 0.79669136 -0.04131459 -0.4278553 -0.24815013 1.116438 -0.026439136 0.354995 0.86839557 -0.68594515 0.19245332 0.33331525 -0.13659574 -0.04335341; 1.030964 -0.77338785 -0.60938394 -0.3199599 0.9414756 -0.74143624 -0.91034496 0.6074505 -0.54232985 -0.30516297 0.21154629 -0.5275524 0.6961487 -0.39980128 -0.47098938 0.5249121 -0.10492691 -0.3207378 -0.6838264 -0.7803789 0.32044253 -0.448465 0.43438867 0.000982338 0.5326028 -0.9711814 -0.17330371 -0.48190466 -0.09707207 0.7035897 -0.64195204 -0.57738703 -0.54431146 0.33509657 -0.24370125 -0.6917539 0.07690157 -0.5502152 -0.48056778 -0.35900092 0.50727504 -0.19777529 -0.64305484 -0.24946287 0.24073908 -0.62933946 0.2849696 0.5067657 -0.76282257 -0.08052275 -0.6986124 -1.0108805 0.77055836 0.4309184 -0.65078586 -0.18708324 0.72680277 -0.06916408 -0.2989908 0.48483488 0.30752322 0.94575644 -0.6322748 -0.5623185 -0.24844892 0.80749047 0.7526038 0.5173646 -0.7246457 0.55262154 0.8391701 -0.49537525 -0.675754 -0.60245115 -0.17894368 0.2951758 -0.52511525 -0.8654914 0.8768734 0.9585862; 0.6019006 -0.60537076 -0.66409355 -0.08655706 0.59295595 -0.7356671 -0.5452003 0.5787859 -1.0825679 -0.31961316 0.14678286 -0.5871489 0.7399228 -0.29467624 -0.33700484 0.7334125 0.25223178 -0.12042228 -0.5616173 -0.7493088 0.46151868 -0.18970722 0.28043282 -0.30987304 0.79066163 -0.73123646 -0.17843613 -0.46635437 0.10542296 0.83993375 -0.7728404 -0.41028714 -0.11486306 0.23389883 -0.27087513 -0.6629905 -0.27231857 -0.5715555 -0.18409505 -0.22026548 0.20221332 -0.5685064 -0.663232 -0.36541322 0.17337443 -0.7550816 0.05521666 0.65639585 -0.83467513 -0.07708701 -0.9724655 -0.6598551 0.70646137 0.3466706 -0.70535547 -0.09780296 0.9463569 -0.5069622 -0.5642399 0.6178395 0.29907456 0.7355935 -0.4607237 -0.9483248 -0.4359857 1.0626094 0.8287831 0.67612517 -0.9237049 0.78632617 0.67991126 0.14592406 -0.72758484 -0.755341 -0.13535605 0.2497937 -0.4262162 -0.75899184 0.8012509 0.6399505; 0.30267215 -0.3861378 0.341607 -0.8932591 0.3934429 -0.3222484 -0.13173 0.23778625 -0.19754687 0.85637665 0.5896629 -0.30519 -0.14765039 -0.9562067 -0.5183746 0.22518098 0.63661385 -0.8925694 -0.5322582 -0.4143222 0.64423233 -0.815544 0.7707687 0.9183804 0.29086074 0.015946068 -0.7733236 0.54169226 -0.92484933 0.33644307 -0.5796242 -0.54458284 0.6619667 0.9126392 -0.6201793 -0.38405174 -0.9562473 0.83481675 -0.7468587 -0.9349402 0.8841793 -0.5782625 -0.5291009 -0.640119 0.97347057 -0.265833 0.88266736 0.5705659 -0.47914657 -0.6521345 -0.10296136 0.7483685 0.42574656 0.5876349 0.04590124 -0.670125 0.026803056 -0.83246684 -0.4776708 0.6478246 0.8959475 -0.39602336 -0.5739977 -0.017063709 -0.57905525 0.14011464 0.0063306713 -0.6919984 -0.06265553 0.33888435 0.2689972 -0.87424207 -0.29734015 -0.21450487 -0.7958087 0.9361939 -0.44385326 -0.091751225 -0.43048683 -0.44573665; 0.5916995 -0.70464957 -0.7755386 0.17587921 0.6240368 -0.5590979 -0.5604042 0.93520904 -0.8592285 -0.65204275 -0.08455207 -0.48206574 0.52224517 -0.16054146 -0.3347269 1.0373818 -0.20230107 0.016216643 -0.9667695 -0.7823865 0.016317738 0.11156946 -0.35421762 -0.5381399 0.8769179 -0.6764597 0.19411154 -0.6183795 0.5864485 1.0666509 -0.6046831 -0.5650915 -0.2891851 -0.43646592 -0.13604803 -0.7846909 0.0104408 -0.6171554 -0.06262191 -0.059272945 0.17229311 -0.31348637 -0.7957886 -0.2287753 -0.39027786 -0.42674634 -0.42288888 0.7774889 -0.83855194 0.2565923 -0.7974151 -0.22585651 0.60026914 0.2994784 -0.6749067 0.26146185 0.9768411 -0.08276019 -0.87265235 0.3821485 0.094735846 0.5185477 -0.28586644 -0.9804604 0.016485972 1.0006342 0.944537 0.8485642 -0.8311003 0.9767073 1.0506049 0.7158243 -1.1422727 -0.76359576 0.199464 -0.3562544 -0.5199233 -0.6805886 0.593682 0.7423914; -0.18290219 0.23487315 -0.38698968 0.9370686 -0.20376514 0.30242267 0.3849416 -0.35627133 0.14664306 -0.9755795 -0.5636456 0.59518605 0.30161208 0.88692474 0.72180516 -0.3352703 -0.79973525 0.7410308 0.29699507 0.3181566 -0.50667465 0.8487277 -0.8858017 -0.9635871 -0.4430649 0.26241043 0.92072487 -0.7694756 0.7326618 -0.3303701 0.503409 0.67920136 -0.64920723 -0.8205949 0.644612 0.20205699 0.8785572 -0.61093473 0.8252194 0.7749175 -0.8318571 0.84393203 0.37899566 0.6611387 -0.9387861 0.42378405 -0.99695355 -0.3039155 0.50057375 0.92768735 0.13825719 -0.49345788 -0.4693738 -0.7770712 -0.25831932 0.6561806 -0.15987918 0.6776218 0.5654713 -0.52447283 -0.8618739 0.17504668 0.5322274 0.13166969 0.65990907 -0.28426272 -0.20786172 0.67979187 0.14833756 -0.34295782 -0.4465821 0.85221404 0.42545846 0.20294867 0.8886171 -0.892209 0.5958478 0.3764483 0.469039 0.3794259; 0.7635995 -0.85921264 -0.6084859 0.4940981 0.7075824 -0.7650322 -0.75454164 0.68493056 -0.73507875 -0.5245042 -0.391912 -0.4992237 0.58236516 0.19357578 -0.43192708 1.0265105 -0.6167501 0.15542485 -0.84372246 -0.7588121 -0.07669378 0.5999773 -0.5402197 -0.37741473 0.89863765 -0.8664136 0.34223658 -0.45051682 0.4132853 0.7245844 -0.65868217 -0.33998454 -0.6567245 -0.6026449 0.19200741 -0.7720922 0.59232384 -0.59984297 0.20511723 0.55570966 -0.31057417 0.2747522 -0.5652698 0.27852342 -0.62209266 -0.813327 -0.69337815 0.8605848 -0.6998423 0.5858783 -0.9323919 -0.6744087 0.5814683 0.14342482 -0.7190475 0.5491573 0.6689398 0.32147905 -0.9390384 -0.14398848 -0.3491998 0.6303374 0.07312081 -0.78101134 0.43900532 0.64356977 0.5855202 0.72172666 -0.6290638 0.6339062 0.8427611 0.5113576 -0.9004863 -0.6925275 0.6039224 -0.36173093 -0.32234588 -0.7584679 0.6887733 0.5736817; 0.5531125 -0.90833956 -0.633321 -0.24006093 0.59976286 -0.6247279 -0.582626 0.6425684 -1.0565565 -0.52907175 0.19997671 -0.6140625 0.83679837 -0.25710228 -0.6704748 0.95180523 0.25614643 -0.38034338 -0.84459776 -0.6482445 0.34356314 -0.2807941 0.034079004 -0.20040895 0.84389603 -0.923024 -0.342797 -0.5875467 -0.07826073 0.8280724 -0.7580639 -0.51287204 -0.08143991 0.1111705 -0.39828232 -0.68441296 -0.06938424 -0.43294045 -0.18307932 -0.42270508 0.25793958 -0.2930857 -0.80940914 -0.41133025 0.027765287 -0.48568952 0.079699986 0.6308153 -0.62986875 0.097427145 -1.0217931 -0.35060397 0.55701613 0.73482764 -1.012061 0.043275584 0.7838117 -0.50066894 -0.74701875 0.43684348 0.27489477 0.6010579 -0.37323314 -0.83378977 -0.5861218 1.0222197 0.94477016 0.8050814 -1.0414231 0.90389574 0.7679905 -0.0715364 -0.9135348 -0.66034824 0.051208686 0.018684402 -0.39912078 -0.94352335 0.44297048 0.80890894; -0.6648467 0.58693737 0.56812346 0.28469896 -0.8323506 0.40288585 0.72705007 -0.6025041 0.66213965 -0.58616364 -0.51612604 0.48144072 -0.84505254 0.49832103 0.47917035 -0.36602932 -0.516843 0.26652986 0.5912622 0.5415608 -0.35174164 0.39088777 -0.5847936 -0.6686746 -0.6586738 0.7688334 0.544361 -0.5393924 0.26246586 -0.5863934 0.6133638 0.5000736 -0.40826988 -0.5599998 0.58502823 0.45087856 0.6607825 -0.3616576 0.34016097 0.55058366 -0.4238151 0.3814713 0.6326835 0.3180326 -0.3532842 0.6591285 -0.4475968 -0.3933881 0.817936 0.44859737 0.49489477 -0.53161454 -0.7790984 -0.45320886 0.6247015 0.29832196 -0.91392964 0.69907635 0.47943923 -0.55108356 -0.53124785 -0.6248919 0.39225566 0.64321667 0.49389 -0.6197821 -0.7759228 -1.1958001 0.84255993 -0.75906485 -0.8057311 0.40430063 0.7522536 0.6049675 0.5999831 -0.31897107 0.49409065 0.7802622 -0.62493783 -0.67133814; -0.7087338 0.13682234 0.36190888 -0.77199066 -0.29915792 -0.24508655 0.6295207 0.4203596 0.15317754 0.7609129 0.5389632 -0.3654038 -0.55869687 -0.76205534 -0.85434675 0.45974308 0.71314675 -0.66232276 0.2144997 -0.14785263 0.89068264 -0.55512494 0.88089293 0.6990165 0.25300753 0.4216314 -1.0044807 0.76170176 -0.67105716 -0.3144712 -0.19232164 -0.80117315 0.6085463 0.5765879 -0.853644 0.3373408 -0.64305747 0.5093555 -0.6804221 -0.97494817 0.59640235 -0.85122615 0.0981653 -1.0285541 0.8995785 0.17588732 0.54979056 0.179429 0.5106376 -0.7539617 0.39942166 0.7377276 -0.44472086 1.044302 0.5105638 -0.91329205 -0.55428845 -0.86218685 -0.57599753 1.0376816 0.9266411 -0.54010344 -0.623338 0.17443804 -0.9262325 -0.46062148 -0.42504302 -0.506296 0.5316339 -0.13548298 -0.2214996 -0.48125848 0.21740754 0.1457518 -0.96044725 0.59054077 -0.4230229 0.6022862 -0.5858204 -0.64510113; -0.21178943 -0.14412804 0.40710542 -0.9419945 0.17018874 -0.5893167 0.15473224 0.21991709 0.11777375 0.82594824 0.8423563 -0.61636287 -0.3276676 -0.7963858 -0.797725 0.2768089 0.67332804 -0.94472784 -0.03653904 -0.40416718 0.7141836 -0.94086254 0.6730361 0.7467601 0.29334417 0.060519595 -0.8170989 0.7314331 -0.876122 0.009726279 -0.20667692 -0.66824377 0.5567519 0.8017883 -0.89368194 -0.050799478 -0.6273713 0.7360689 -0.5578735 -0.7661158 0.6637087 -0.9655133 -0.3056035 -0.71596533 0.9674932 -0.09337283 1.0296257 0.48744348 0.046798564 -0.7469278 0.05341871 0.27469692 0.10335645 0.55935067 0.6047173 -0.65089625 -0.3013538 -0.8120626 -0.5208095 0.9602427 0.95125186 -0.6903527 -0.7677143 -0.11267999 -0.75903815 -0.047464743 0.016479248 -0.67686355 0.487351 0.17346485 0.19527592 -0.7155787 -0.019663021 0.119717315 -0.84154063 0.88776225 -0.6158212 0.2391493 -0.61899555 -0.47061202; 0.12080787 -0.44348767 0.23117061 -0.9409497 0.3294125 -0.6249977 -0.24589205 0.5474728 -0.21864001 0.9224936 0.8211664 -0.5894534 -0.4507793 -0.8882084 -0.48719016 0.21397933 0.81689376 -0.7810914 -0.16742904 -0.5053584 0.72706085 -0.76160073 0.89597666 0.9342959 0.44508362 -0.013819161 -0.83608663 0.81017566 -0.87741536 0.20798908 -0.33584014 -0.5699923 0.64690846 0.62092423 -0.7850818 -0.2623039 -0.6861033 0.63864803 -0.77172595 -0.9889144 0.87610257 -0.8462547 -0.22990035 -0.923522 0.7081555 -0.4515503 0.8936667 0.24601018 -0.23497808 -0.95725363 0.014257987 0.39783412 0.022422355 0.66016084 0.31890747 -0.79338306 -0.031715762 -0.5572842 -0.42903826 0.8182179 0.75078636 -0.33351216 -0.7018182 -0.10822428 -0.6792068 0.23140162 0.0050405725 -0.72992456 0.12109493 0.27346542 0.42026246 -0.69947886 -0.18741252 -0.23913388 -1.0182527 0.8433455 -0.38514623 -0.06029913 -0.30195147 -0.33300954; -0.8059172 0.92447823 0.7010604 -0.17203452 -0.76575965 0.7502314 0.66134894 -0.63007534 0.9518231 0.55586874 0.123014055 0.665393 -0.52124447 0.13843004 0.48261297 -0.8132437 0.13277934 0.033053447 0.5468399 0.8628131 -0.11826804 0.14024122 0.1686046 0.024175012 -0.91660297 0.91852456 -0.0030428106 0.309441 -0.25273982 -0.92461294 0.48904493 0.48073307 0.4837945 0.28168476 0.3598708 0.8543303 0.14078212 0.67760026 0.30684927 0.29699782 -0.0660411 0.37772676 0.899721 0.5278647 0.13011447 0.47169513 0.32032517 -0.7570112 0.75884074 -0.054861918 1.0471075 0.2306638 -0.58936286 -0.4890144 0.7286322 -0.3535996 -1.1236669 0.10423337 0.9175323 -0.30741975 -0.41003242 -0.5376945 0.19409147 0.823824 0.35827604 -0.823681 -1.1683557 -0.63347936 1.1419388 -0.83417505 -0.8179336 -0.3043449 0.92571086 0.8733958 -0.24440934 0.22401568 0.64374816 0.80001074 -0.57662934 -0.38568977; -0.53016394 0.59121275 0.69252235 0.43059334 -0.534902 0.8883448 0.7178598 -0.39145926 0.63410616 -0.052716922 -0.3259263 0.68206304 -0.7505284 0.60519516 0.4273585 -0.804725 -0.38226178 0.5273743 0.78085524 0.7106594 -0.518633 0.468095 -0.51278794 -0.14711048 -0.4803187 0.8742196 0.17909479 0.6626393 0.45353967 -0.6830751 0.64593166 0.2489147 0.019000702 -0.5323597 0.26002955 0.75035703 0.48491418 0.47410622 0.46887362 0.3763395 -0.46964464 0.4020179 0.70717406 0.41533014 -0.47695935 0.8428542 -0.39458367 -0.5370187 0.7148921 0.34630248 0.7340027 0.8664492 -0.8642976 -0.5474497 0.8605124 0.30969286 -0.6000852 0.22590068 0.29165328 -0.3925836 -0.27585727 -0.9172611 0.34066638 0.54292905 0.34698302 -0.78011507 -0.61322093 -0.028948208 0.8332906 -0.5622186 -0.6497278 0.49492392 0.5169448 0.77486205 0.20546353 -0.3811106 0.62694275 0.58496344 -0.7798618 -0.8249669; 0.65619636 -0.6894287 -0.70372206 -0.065983 0.676506 -0.8808338 -0.8896672 0.60403943 -0.6696552 -0.31729078 0.0255679 -0.7905689 0.72620046 -0.47737616 -0.5339856 0.58825403 -0.3788131 -0.4918596 -0.6509455 -0.6821059 0.3962178 0.11833996 -0.03269948 -0.25593266 0.6495052 -0.8404903 0.26639426 -0.39439166 0.51407236 0.872403 -0.5852919 -0.34404027 -0.5362301 -0.14645375 -0.31227726 -1.034097 0.101462044 -0.4150724 -0.098292105 -0.19021443 0.22350392 -0.22639236 -0.7111111 -0.3683983 -0.051712114 -0.8153334 -0.13772339 0.7548588 -0.72409743 0.14387783 -0.87653047 -0.47332156 1.0116293 0.53470135 -0.67052335 0.40490705 0.88644636 -0.017119203 -0.6803082 0.39933816 0.19307163 0.8403013 -0.47353205 -0.97227037 -0.29524025 0.8300561 0.9204111 0.5319965 -0.77028036 0.8966802 0.92254364 0.26861653 -0.9226706 -0.7618148 0.16815215 -0.16975546 -0.3318168 -0.6487937 0.60490704 0.52577174; -0.7751391 0.66169363 0.71479154 -0.6315582 -0.6809003 0.8503859 0.63276494 -0.8893876 0.6517583 0.6012286 0.45448476 0.87892365 -0.59743506 -0.22827865 0.32029805 -0.9886821 0.6136972 -0.22854559 0.73746765 0.71151465 0.18031444 -0.41266778 0.4961218 0.48635003 -0.8283229 0.8254417 -0.4784671 0.5732909 -0.65304655 -0.7939162 0.81679696 0.13866772 0.6786653 0.58286965 -0.3771562 0.57270247 -0.5360719 0.44865912 -0.051516186 -0.21772644 0.20038457 -0.3381061 0.8539639 -0.095951 0.6170539 0.6081036 0.69147235 -0.6357626 0.716251 -0.6385817 0.70109606 0.5450045 -0.6711878 -0.16222222 0.6585692 -0.40000072 -0.7418796 -0.22120263 0.8809312 0.1433323 0.28698328 -0.45158345 -0.03448282 0.9240474 -0.4738788 -0.75125563 -0.62608093 -0.7873304 0.8175445 -0.7479149 -0.66367036 -0.6208549 0.71298105 0.8696593 -0.5825649 0.40442237 0.6218299 0.606342 -0.5715685 -0.50779927; -0.7400746 0.80833083 0.6368615 -0.46905935 -0.78446627 0.95859444 0.89562756 -0.5585828 0.7098652 0.65125155 0.40217605 0.740095 -0.7974041 0.12479544 0.56227565 -0.82954144 0.42238015 0.09427937 0.67571527 0.9442145 0.08715031 -0.14643146 0.41654852 0.268021 -0.9725149 0.6143674 -0.35470062 0.47077882 -0.5316606 -0.77510506 0.6160868 0.47631493 0.6576166 0.17588043 -0.019702595 0.6391721 -0.35231623 0.5391125 -0.058883023 -0.23042566 -0.05103454 -0.18976243 0.6822728 -0.040626258 0.41524717 0.5917906 0.49331945 -0.8373796 0.878746 -0.43717158 0.94834673 0.31466824 -0.6780933 -0.45995972 0.828181 -0.35516977 -0.7899343 -0.32845685 0.7906426 -0.29087707 0.12801841 -0.45526803 0.04861406 0.83408904 -0.10511011 -0.9628615 -0.9898377 -0.6449202 0.7037238 -0.8979248 -1.0135782 -0.39098465 0.73713684 0.7299575 -0.29519388 0.5496928 0.611116 0.5998028 -0.5218185 -0.44532868; -0.19358967 -0.15941162 0.42291963 -0.700127 0.20405072 -0.35755864 -0.2618009 0.24986978 -0.29680783 0.79591423 0.6580222 -0.4531668 -0.44990227 -0.56319517 -0.52770895 0.6020825 0.66331685 -0.61346483 -0.15979588 -0.44211775 0.5962425 -0.8740929 0.65198535 0.9912686 0.42985493 -0.15638177 -0.9753791 0.6378892 -0.7217815 0.02023627 -0.3316051 -0.84254736 0.9474795 0.8402222 -0.85453427 -0.14609197 -0.69891685 0.6813278 -0.8250432 -0.9381781 0.77480775 -0.8434479 -0.10903676 -0.5994443 0.68939424 -0.16505657 0.8712212 0.30763364 -0.28354517 -0.9545514 0.081945084 0.49511045 0.04531813 0.7122771 0.21180952 -0.7941162 -0.16818644 -0.80687684 -0.5143768 0.5375406 0.8559531 -0.40620497 -0.66508645 -0.00675543 -0.712381 0.06546242 0.015385094 -0.41696146 0.035655614 0.39025533 0.35181203 -0.88285035 -0.32361495 -0.20373571 -0.6938189 0.8601408 -0.4515854 -0.15967856 -0.47525245 -0.49495968; -0.1971823 0.3888159 -0.53242177 0.75245035 -0.28776547 0.3114837 0.27773234 -0.37719902 0.15133001 -0.7319241 -0.8995866 0.42255914 0.44663346 0.5734549 0.5281589 -0.3833614 -0.90593725 0.73898906 0.22959179 0.23417921 -0.74108636 0.7976405 -0.69846576 -0.80562234 -0.59748554 0.034184515 0.95646876 -0.5331277 0.935306 -0.35779998 0.25629783 0.7210579 -0.96208805 -0.591116 0.6142628 0.2675558 1.007791 -0.48164392 0.6221199 0.7984877 -0.73310095 0.98073655 0.25543392 0.6088024 -0.73956597 0.18908206 -0.89658386 -0.56206715 0.14933716 0.7013913 0.33056065 -0.84460896 -0.20025498 -0.55460244 -0.16705899 0.8840263 0.0956532 0.8546115 0.3507982 -0.73072577 -0.8061489 0.4019861 0.5863937 0.44002888 0.7153442 0.0060531665 -0.11278623 0.6356425 -0.1290332 -0.14517187 -0.20037739 0.6871827 0.20053415 0.12949589 0.8860668 -0.7729454 0.6417057 0.026758758 0.4030443 0.21437444; -0.50794655 0.10321189 0.5224344 -0.7432396 -0.35238022 -0.42170146 0.44569403 0.4027979 0.4134007 0.62146175 0.63896084 -0.5325934 -0.65711695 -1.0498922 -0.5569712 -0.024138838 0.63832015 -0.8748888 0.15231913 -0.17326982 0.78440815 -0.8861248 0.64741397 0.52003783 0.31645793 0.4207789 -0.6090366 0.50424105 -0.6650336 -0.15491742 -0.14490448 -0.89081436 0.60828507 0.7454338 -0.86644936 0.40274388 -0.7347309 0.53249156 -1.0233802 -0.8231183 0.9940789 -0.8411582 0.06494707 -0.92587817 0.63156384 0.2664003 0.6323404 0.30260628 0.28275532 -0.7792144 0.57773477 0.3622727 -0.49169743 0.63887686 0.52775025 -0.54675764 -0.38456434 -0.69543767 -0.5254578 0.88604766 0.98637074 -0.59534323 -0.6604926 0.4587188 -0.97196525 -0.2973483 -0.4266452 -0.44106436 0.54724324 -0.29275033 -0.0015977549 -0.5944168 0.3037606 0.13141236 -0.60120255 0.72883517 -0.76125205 0.4237632 -0.38049883 -0.68278927; 0.6218826 -0.83212614 -0.5086869 0.19171618 0.8551936 -0.5918643 -0.568541 0.7905533 -1.05727 -0.60368055 -0.06629578 -0.70558834 0.6081802 -0.37048364 -0.32526165 0.8532164 -0.2521162 -0.098670445 -0.73867476 -0.6849531 0.34800136 0.08534798 0.03474804 -0.25851336 0.96772707 -0.8789787 0.13793424 -0.4711199 0.5141093 1.0498524 -0.5580194 -0.48702613 -0.36220255 -0.12787053 -0.4417608 -0.84199876 0.3660327 -0.5453487 -0.20192896 -0.15410027 0.094916455 -0.26393706 -0.7538032 -0.18325336 0.024596466 -0.78663117 -0.23701183 0.62126064 -0.6415784 0.15704997 -0.76984614 -0.45334774 0.64918363 0.38249514 -0.7041773 0.4501805 0.90526444 -0.10041196 -0.58462787 0.4963481 0.3588022 0.66317147 -0.15565117 -1.0078995 -0.22187182 1.0248125 0.8064215 0.7690471 -0.91827327 0.9160186 0.8936477 0.391044 -0.8970397 -0.9042758 0.1707686 -0.3482118 -0.32888693 -0.90075636 0.6786819 0.63948673; 0.57604456 -0.8177164 -0.55419344 0.44524702 0.5886902 -0.7753788 -0.6729239 0.7056304 -0.64399046 -0.6086391 -0.5552341 -0.8058657 0.63129145 0.23867108 -0.35842448 0.96260786 -0.4496948 0.13214411 -0.71948075 -0.7980414 -0.40200153 0.5292948 -0.6412586 -0.54223347 0.7055953 -0.6588357 0.39199033 -0.47397044 0.5658999 0.7829984 -0.99124765 0.15129332 -0.52548313 -0.57089496 0.19436884 -0.6883398 0.40368596 -0.6678803 0.49201092 0.40219855 -0.3917191 0.30455592 -0.8243644 0.37782148 -0.41076085 -0.50464475 -0.5372966 0.6165259 -0.74116397 0.47910675 -0.5507306 -0.52340984 0.7019364 -0.10547721 -0.59342045 0.54708725 0.6400475 0.2834988 -0.8732295 -0.40894228 -0.409871 0.61630815 0.037465163 -0.82862645 0.5676493 0.61274564 0.54207873 0.5483844 -0.5977104 0.75294584 0.71909755 0.5460927 -0.8718135 -0.7686287 0.35082716 -0.34176213 -0.7050034 -0.53266 0.40960136 0.71472746; 0.9254428 -0.53095365 -0.93156475 -0.29450724 0.7272342 -0.7565288 -0.8714679 0.54820836 -0.46356314 0.47890586 0.6908432 -0.6000685 0.9822379 -0.52684575 -0.4908121 0.543623 0.32028553 -0.6532476 -0.7119424 -0.5618683 0.3712483 -0.51344347 0.6110984 0.21735887 0.4089272 -0.8217244 -0.26582643 -0.3074255 -0.22177723 0.50101036 -0.8893829 -0.28184676 0.3639827 0.55882317 -0.24218452 -0.85221153 -0.33080935 -0.3766631 -0.3585997 -0.22268482 0.3793751 -0.32862946 -0.73419076 -0.27881753 0.63408685 -0.8225395 0.6079581 0.710251 -0.6607453 -0.38940567 -0.5180214 -0.99658376 0.7849514 0.38006514 -0.8619632 -0.20406632 0.49455822 -0.30193704 -0.22679067 0.5182144 0.3774387 0.80121285 -0.5746615 -0.70902014 -0.30791494 0.7125717 0.5329322 0.08405505 -0.5789403 0.61596966 0.39265257 -0.5143214 -0.4365206 -0.80365807 -0.22719818 0.23947757 -0.50217265 -0.8396634 0.9264985 0.9178347; 0.565998 -0.9409974 -0.7916091 0.3156616 0.6581142 -1.0076743 -0.73356616 0.63624007 -0.82072693 -0.53021073 -0.18478304 -0.69926846 0.80711526 -0.168291 -0.45721316 0.7014214 -0.63824505 -0.14139737 -0.950652 -0.88776946 -0.065348655 0.44342968 -0.35361934 -0.4346258 0.72669196 -0.86063915 0.4466292 -0.7114083 0.44970885 0.89590764 -0.8926207 0.011435239 -0.7689262 -0.16532604 0.055467952 -0.58120507 0.34143177 -0.6039217 0.017706878 0.30891538 0.118615836 0.29272014 -0.8112651 0.37498677 -0.18715394 -0.7815926 -0.48533145 0.72415805 -0.60275817 0.6148435 -0.8242696 -0.6379551 0.93375605 0.2087073 -0.6236589 0.52612746 0.8144766 0.5502634 -0.60137355 0.08130516 -0.22387794 0.6437488 -0.16942288 -0.88551784 0.2151426 0.7577829 0.64909023 0.7975612 -0.43992603 0.61394036 0.9344765 0.35462514 -0.93534386 -0.78988105 0.51253366 -0.48264787 -0.36213127 -0.83310586 0.796849 0.56679994], bias = Float32[-0.015529476; -0.7256059; 0.7168833; 0.6703226; -0.78362644; 0.03901989; -0.43658847; 0.42249113; -0.5797554; -0.8413509; 0.7846166; -0.7712585; -0.36294755; 0.21753253; 0.5137484; 0.14588177; 0.015424962; -0.37022847; 0.5253725; -0.5061441; -0.03545559; 0.6279705; 0.7975671; -0.07034765; -0.7673932; 0.8703869; 0.43890554; 0.41790956; 0.85832083; 0.8036127; 0.12697889; -0.81352025; -0.80967337; -0.35362685; 0.30065775; 0.3216815; -0.42059505; 0.8372364; -0.35243806; 0.5543911; -0.43236798; 0.7637857; -0.79004836; -0.055746518; -0.64780426; 0.30059028; -0.78892076; 0.4899541; 0.76225364; -0.8091825; -0.47567505; 0.70398945; -0.37966156; -0.8515585; -0.70479923; -0.8457195; 0.97153926; -0.3629585; -0.14720069; -0.801746; 0.1912806; 0.74687934; 0.46532282; -0.109616056; 0.44107312; -0.6506963; -0.8161442; -0.7939894; -0.16530022; 0.45920107; 0.016204253; -0.48918602; -0.3563699; -0.81506795; 0.7727356; -0.76895285; 0.14332858; 0.5655532; -0.53221864; 0.38117263]), layer_3 = (weight = Float32[-0.36457172 0.32865506 0.42765424 -0.78771806 0.63584244 -0.2603471 0.14444758 -0.35100487 -0.50644755 0.00091739465 -0.707 0.123417504 0.7023612 0.55260587 -0.808067 -0.23636925 -0.3020036 -0.24952747 0.39496347 0.31662 -0.77066416 0.42969447 0.1042431 -0.30155578 1.0264097 0.5642662 0.033212103 -0.6217573 0.62019426 -0.080962986 -0.66309786 0.40914822 0.3894719 -0.0069380375 -0.054287013 0.48526374 -0.10005282 -0.3121228 0.35228106 -0.48776776 0.19216935 -0.40806514 0.396859 0.5856543 -0.56556916 0.19708242 0.7373318 -0.8860318 -1.0861267 0.16851766 0.8805488 0.33572644 -0.26570243 0.08266941 1.0561284 -0.62531304 -0.9457761 -0.20937069 0.044389073 0.66675574 -0.6839062 -1.0285156 -0.8851651 -0.64857024 0.2236508 0.7068124 0.34066415 1.055493 0.56281954 0.5870524 0.5000695 0.90589 0.43875074 -0.625369 0.07399367 0.06228555 -0.14581169 0.5597075 -0.5798255 -0.09079072; -0.5336554 -0.34984714 -0.3178965 0.15193488 -0.7431177 0.44247657 0.4111847 -0.5652835 -0.2542314 0.7606675 0.8020022 -0.8967685 0.70413595 0.29196802 0.3034787 0.5203308 -0.41783252 -0.8677605 0.4006184 -0.55322295 0.025880788 -0.10065168 0.8395966 0.74688554 -0.43419805 -0.21478166 1.3714609 -0.5993328 -0.13416529 0.70060664 0.9618887 -0.7323608 -0.8315892 -1.1040561 0.6622035 0.4649551 0.51154065 0.63709915 0.5040368 0.46360725 -1.3400041 -0.90947 -0.7049043 -0.19134817 0.14979683 -0.74813735 -0.39708427 0.21935004 0.014403835 -0.7935353 0.047432814 0.47396827 0.84487116 0.6544158 -0.20834602 0.25666162 0.5824046 0.6098791 0.53852767 -0.79868615 0.40097347 0.039610982 0.11662093 -0.052758295 -0.60800916 0.6574299 -0.7315227 -0.10891027 -0.076255165 -0.10557638 0.5051376 -0.1347601 -0.45114315 -0.0025640714 -0.7111825 -0.6621846 0.47838256 0.25199118 -0.008250485 -1.0045863], bias = Float32[-0.7922723; 0.21831562]))

#Neural ODE prediction implementation
prob_node_extrapolate = ODEProblem(dudt_node,I, etaspan2, p_trained)
_sol_node = solve(prob_node_extrapolate, Tsit5(),saveat = collect(etasteps2))
#Neural ODE Extrapolation scatter plot
p_neuralode = scatter(_sol_node, legend = :topright,markeralpha=0.5, label=["NeuralODE \\phi" "NeuralODE \\phi'"], title="Neural ODE Extrapolation")

#Trained (predicted) DATA up to the 90 elements with the Neural ODE, retrieving. Same as callback.
p=result_neuralode2.minimizer
p = p_trained
prob_neuralode = NeuralODE(dudt2,etaspan; saveat = etasteps)
predicted=(prob_neuralode(I, p, st)[1])

#Plot
scatter!(collect(etasteps), predicted[1, :],color=:black,markershape=:hline; label = "\\phi prediction")
xlabel!("\\eta")
scatter!(collect(etasteps), predicted[2, :],color=:black,markershape=:cross; label = "\\phi ' prediction")
savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0\\Neural ODE\\Results\\NoNoise\\Whitedwarf_NODEpredictionvspredictedtrainingdata.png")



#Ground truth full data vs Neural ODE full prediction

p_neuralode = scatter(_sol_node, legend = :topright,markeralpha=0.5, label=["NeuralODE \\phi" "NeuralODE \\phi'"], title="Neural ODE Extrapolation")

plot!(sol, linewidth = 1, xaxis = "\\eta",
     label = ["\\phi ODE" "\\phi' ODE"])



savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0\\Neural ODE\\Results\\NoNoise\\Whitedwarf_no_noise_NODEvsGroundTruthData_ODE.png")

#Final plot for the results- better formated
plot(sol.t[1:end-10],Array(sol[:,1:end-10])[1,:],color=:blue, linewidth = 1, xaxis = "\\eta",
     label = "Training \\phi ", title="White Dwarf model")

plot!(sol.t[1:end-10],Array(sol[:,1:end-10])[2,:],color=:blue, linewidth = 1, xaxis = "\\eta",
     label = "Training \\phi'")
xlabel!("\\eta (dimensionless radius)")

#Trained Phi NODE
scatter!(collect(etasteps), predicted[1, :],color=:blue,markeralpha=0.3; label = "Predicted \\phi")

scatter!(collect(etasteps), predicted[2, :],color=:blue, markeralpha=0.3;label = "Predicted \\phi'")
scatter!(sol.t[end-9:end],_sol_node[1,end-9:end],color=:orange,markeralpha=0.6,label="Forecasted \\phi")

scatter!(sol.t[end-9:end],_sol_node[2, end-9:end],color=:orange,markeralpha=0.6,label="Forecasted \\phi'")

savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0\\Neural ODE\\Results\\NoNoise\\Whitedwarf_forecasted_model.png")

#Second version

plot(sol.t[1:end-10],Array(sol[:,1:end-10])[1,:],color=:blue, linewidth = 1, xaxis = "\\eta",
     label = "Training \\phi ", title="White Dwarf model")

plot!(sol.t[1:end-10],Array(sol[:,1:end-10])[2,:],color=:blue, linewidth = 1, xaxis = "\\eta",
     label = "Training \\phi'")
xlabel!("\\eta (dimensionless radius)")

#Trained Phi NODE
scatter!(collect(etasteps), predicted[1, :],color=:blue,markeralpha=0.3; label = "Predicted \\phi")

scatter!(collect(etasteps), predicted[2, :],color=:blue, markeralpha=0.3;label = "Predicted \\phi'")
scatter!(sol.t[end-9:end],_sol_node[1,end-9:end],color=:orange,markeralpha=0.6,label="Forecasted \\phi")

scatter!(sol.t[end-9:end],_sol_node[2, end-9:end],color=:orange,markeralpha=0.6,label="Forecasted \\phi'")
scatter!(sol.t[end-9:end],Array(sol)[1,end-9:end],color=:black, markershape=:hline,label="ODE \\phi")

scatter!(sol.t[end-9:end],Array(sol)[2,end-9:end],color=:black, markershape=:cross,label="ODE \\phi'")
savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0\\Neural ODE\\Results\\NoNoise\\Whitedwarf_forecasted_model_ODEVersion.png")




#Last Version for the preprint

#----------------------------------

scatter(sol.t[1:end-10],Array(sol[:,1:end-10])[1,:],color=:blue,markeralpha=0.3, linewidth = 1, xaxis = "\\eta",
     label = "Training \\phi ", title="White Dwarf model")


scatter!(sol.t[end-09:end],Array(sol[:,91:end])[1,:], color=:red,markeralpha=0.3, label = "Testing \\phi")

plot!(sol.t[1:end-10],predicted[1, :],color=:blue,markeralpha=0.3; label = "Predicted \\phi")
xlabel!("\\eta (dimensionless radius)")

plot!(sol.t[end-10:end],_sol_node[1,end-10:end],color=:red,markeralpha=0.30,label="Forecasted \\phi")
title!("Trained Neural ODE")
savefig("C:\\Users\\Raymundoneo\\Documents\\SciML Workshop\\bootcamp\\WhiteDwarf_Forecasting_from0\\Neural ODE\\Results\\NoNoise\\NeuralODEModel_finalversion.png")

